#include "mtx3x52.h"
#include "c:\Program Files\National Instruments\MeasurementStudio\CVI\instr\MTX3x52\mtx3x52.H"
#include <userint.h>
#include <utility.h>
#include <formatio.h>
#include <visa.h>		  
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
static ViReal64 toto[50000];

/*==================================================================================*/
#define MTX3x52_REVISION   "Rev 1.0, 04/2005,CVI 6.0"/* Instrument driver revision  */
#define BUFFER_SIZE        512L                      /* File I/O buffer size        */
#define INVALID_SAMPLE     0xFFFF                    /* Sample validity criterion   */
#define MAX_MEM_SIZE       200300                    /* Memory needed for file I/O  */
#define INVALID_VALUE      1E100                     /* Invalid sample default value*/
#define TMO_VALUE          4000                      /* Time out value              */
#define TMO_TRANSFER       20000                     /* Time out value              */
#define HEADER_SIZE        200                       /* Size of binary data header  */
#define HEADER_FILE        20                        /* Size of binary file header  */
/*==================================================================================*/

/*=MTX3x52====================================================================*/
/* LabWindows/CVI 4.01 Instrument Driver (VISA I/O)                          */
/* Original Release: April 1998                                              */
/* By: Jrme Manera (Metrix, Annecy, France)                                */
/*                                                                           */                           
/* Metrix Technical Support: Olivier Lemaire                                 */
/*      PH. (33) 04 50 64 22 22     Fax (33) 04 50 64 22 99                  */
/*                                                                           */
/* For LabWindows Technical Support in the United States call:               */
/*              National Instruments, Austin, Texas                          */
/*              Ph (800) 433-3488    Fax (512) 794-5678                      */
/*                                                                           */
/*                                                                           */
/* Modification History: None                                                */
/*                                                                           */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/
/*-Vertical command arrays---------------------------------------------------*/
static ViString cmdVertCoupling[] = {"AC","DC","GRO"};       
static ViString cmdVoltDiv[] = {"20mV","40mV","80mV", "160mV","400mV","800mV",
						"1.6V","4V","8V", "16V","40V","80V", "160V","400V","800V",
						"1.6kV","4kV","8kV"}; 
static ViReal64 cmdOffsetMax[] = {80E-3,80E-3,400E-3,400E-3,2.0,2.0,2.5,12.5,12.5,
						64.0,64.0,64.0,640.0,640.0,640.0,6400.0,6400.0,6400.0}; 
static ViString cmdInpSelect[]  = {"CHAN","MEM"};   

/*-Horizontal command arrays-------------------------------------------------*/
static ViString cmdTimeDiv[] = {"1E-9", "2E-9", "5E-9", "10E-9", "20E-9", "50E-9",
								"100E-9", "200E-9", "500E-9", "1E-6", "2E-6", "5E-6", "10E-6", "20E-6",
						"50E-6", "100E-6", "200E-6", "500E-6", "1E-3", "2E-3", "5E-3",
						"10E-3", "20E-3", "50E-3", "100E-3", "200E-3", "500E-3","1",
						"2" ,"5", "10", "20", "50", "100", "200"};
                        
static ViReal64 cmdOffset[] = {1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9, 1E-9,
							1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6, 1E-6,
							1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3, 1E-3,
							1,    1,    1,    1,    1,    1,    1,    1};
static ViString cmdDispTrace[] = {"A", "XY"}; 
static ViString cmdcalcfft[] = {"RECT", "HAMM", "HANN", "BLAC"};
static ViString cmdScale[] = {"LIN", "LOG"};

/*-Trigger command arrays---------------------------------------------------*/
static ViString cmdTrigCoupling[] = {"AC","DC","ON"};       
static ViString cmdSlope[] = {"NEG","POS"};
static ViString cmdSource[]= {"INT1", "INT2", "EXT", "LINE"}; 
static ViString cmdType[] = {"0","1","0"};  

/*-Display command arrays----------------------------------------------------*/
static ViString cmddevice[]={"SCOP", "ANALYS", "REC"};
static ViString cmdDisplayLang[] = {"ENG", "FRE","GER", "SPA", "ITA"};
static ViString cmdDisplayScreen[] = {"0", "15", "30", "60"};
static ViString cmdSystemStandby[] = {"30", "60", "240", "1440"};
static ViString cmdMaxminAcq[] = {"ENV", "NORM"};
static ViString cmdMode[] = {"NORM", "ENV"};       
static ViString cmdAverage[] = {"0", "2", "4", "16", "64"};       

/*-Hardcopy command arrays---------------------------------------------------*/
static ViString cmdDestination[] = {"SER", "CENT", "NET", ""};       
static ViString cmdHardcopyLang[] = {"IBM", "EPS", "CAN", "PCL", "DPU", "POST", "BMP", "GIF"}; 
/*-Read command arrays-------------------------------------------------------*/
static ViString cmdReadMeasure[] = {"MIN", "MAX", "PTP", "LOW", "HIGH", "AMPL", "AC","VOLT",
							"RISE:OVER", "FALL:OVER", "RTIME", "FTIME", "PWID","NWID", "PER", "FREQ", "PDUT",
							"PUL:COUN", "PHAS"};  
static ViReal64 cmdTimeBase[] = {1E-9, 2E-9, 5E-9, 10E-9, 20E-9, 50E-9, 100E-9, 200E-9, 500E-9,
							1E-6, 2E-6, 5E-6, 10E-6, 20E-6, 50E-6, 100E-6, 200E-6, 500E-6, 1E-3,
							2E-3, 5E-3, 10E-3, 20E-3, 50E-3, 100E-3, 200E-3, 500E-3, 1.0, 2.0, 5.0,
							10.0, 20.0, 50.0, 100.0};
static ViReal64 cmdVolt[] = {2.5E-3,5E-3,10E-3,20E-3,50E-3,100E-3,200E-3,500E-3,1.0,
							2.0,5.0,10.0,20.0,50.0,100.0,200.0};
static ViString cmdRef[] = {"INT1", "INT2"};
                                     
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/*  mtx3x52_stringValPair is used in the  mtx3x52_errorMessage function      */
/*===========================================================================*/
typedef struct   mtx3x52_stringValPair
{
   ViStatus stringVal;
   ViPString stringName;
}   mtx3x52_tStringValPair;
	 
struct mtx3x52_statusDataRanges {
    ViInt16 triggerMode;
    ViInt16 val2;
    ViInt16 val3;
    ViChar instrDriverRevision[256];
    ViInt16 instrMode;
};
typedef struct mtx3x52_statusDataRanges * mtx3x52_instrRange;

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean mtx3x52_invalidViBooleanRange (ViBoolean val);
ViBoolean mtx3x52_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean mtx3x52_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean mtx3x52_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean mtx3x52_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean mtx3x52_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean mtx3x52_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  mtx3x52_makeDataReadable (ViByte tabByte[], ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples,
                                ViPReal64 XStart, ViPReal64 XIncrement);
ViBoolean mtx3x52_invalidFilename (ViString filename, ViInt16 drive, ViString type);
ViStatus  mtx3x52_instrStatus (ViSession instrSession);
ViStatus  mtx3x52_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);
ViStatus  mtx3x52_defaultInstrSetup (ViSession openInstrSession);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function:	Initialize                                                   */
/* Purpose:		This function opens the instrument, queries the instrument   */
/*				for its ID, and initializes the instrument to a known state  */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx3x52_init (ViRsrc resourceName, ViBoolean IDQuery,
					ViPSession instrSession)
{
	ViStatus  mtx3x52_status = VI_SUCCESS;
	ViSession rmSession = 0;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte model[2];
	ViUInt16 interface;
    
	/*- Check input parameter ranges ----------------------------------------*/
	if ( mtx3x52_invalidViBooleanRange (IDQuery))
		return VI_ERROR_PARAMETER2;

	/*- Open instrument session ---------------------------------------------*/

	if (( mtx3x52_status = viOpenDefaultRM (&rmSession)) < 0)
		return  mtx3x52_status;

	if (( mtx3x52_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0) 
		{
		viClose (rmSession);
		return  mtx3x52_status;
		}	

	viGetAttribute (*instrSession, VI_ATTR_INTF_TYPE, &interface);

	/* Serial Interface initialized to 115200 bauds, 8 databits, 1 stopbit, no parity, RTS/CTS protocol*/
	if(interface==VI_INTF_ASRL) 
		{
		if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 115200)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_DATA_BITS, 8)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_STOP_BITS, VI_ASRL_STOP_ONE)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_PARITY, VI_ASRL_PAR_NONE)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_RTS_CTS)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viSetBuf (*instrSession, VI_ASRL_IN_BUF|VI_ASRL_OUT_BUF, 5000)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0D)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT, VI_ASRL_END_NONE)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viWrite (*instrSession, "\r", 1, &retCnt)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		}

	/*- Configure VISA Formatted I/O ----------------------------------------*/
	if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR_EN, VI_FALSE)) < 0)
		return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
	if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
	if (( mtx3x52_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
		return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
	if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS)) < 0)
		return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
	if (( mtx3x52_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS)) < 0)
		return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);

	/*- Identification Query ------------------------------------------------*/
	if (IDQuery) 
		{
		if (( mtx3x52_status = viWrite (*instrSession, "*IDN?\r", 6, &retCnt)) < 0)
			return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
		if (( mtx3x52_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
			return  mtx3x52_status;
		if (FindPattern (rdBuffer, 0, -1, "MTX3", 1, 0) == -1)
		return  mtx3x52_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
		}

	/*- Send Default Instrument Setup ---------------------------------*/
																			   
	if (( mtx3x52_status =  mtx3x52_defaultInstrSetup (*instrSession)) < 0)
		return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
	
	if (( mtx3x52_status = viWrite (*instrSession, "FORM INT;FORM:DINT 1\r", 21, &retCnt)) < 0)
		return  mtx3x52_initCleanUp (rmSession, instrSession,  mtx3x52_status);
	
	/*-Check instrument status & Return from function------------------------*/
	if (( mtx3x52_status =  mtx3x52_instrStatus(*instrSession)) < 0)
		return  mtx3x52_status;

	return  mtx3x52_status;
}

/*===========================================================================*/
/* Function: Example                                                         */
/* Purpose:  This function is an example of this driver's functions use.     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_example (ViSession instrSession, ViInt16 channel,
					ViInt16 timebase, ViInt16 voltageRange, ViInt16 coupling, 
					ViReal64 _VI_FAR data[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
    
	/*- Check input parameter ranges ----------------------------------------*/    
	if (mtx3x52_invalidViInt16Range (channel, 0, 1))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range (timebase, 0, 33))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViInt16Range (voltageRange, 0, 17))
		return VI_ERROR_PARAMETER4;
	if (mtx3x52_invalidViInt16Range (coupling, 0, 2))
		return VI_ERROR_PARAMETER5;
    
	/*- Configuring vertical settings----------------------------------------*/
	if((mtx3x52_status = mtx3x52_verticalSetup (instrSession, channel + 1, VI_ON, coupling, 1, voltageRange, 0.0, VI_ON)) < 0)
		return mtx3x52_status;
        
	/*- Setting timebase ----------------------------------------------------*/
	if((mtx3x52_status = mtx3x52_horizontalSetup (instrSession, timebase, -250.0)) < 0)
		return mtx3x52_status;

	/*- Setting trigger mode to AUTO-----------------------------------------*/
	/*- MTX3x52 starts acquiring data immediately after it is triggered-------*/
	if((mtx3x52_status = mtx3x52_setGeneralTrigger (instrSession, channel, VI_ON, 0,VI_OFF, 1, 0.0)) < 0)
		return mtx3x52_status;
        
	/*- Sending single-shot trigger in wait mode-----------------------------*/
	/*- This function will return as long as the acquisition is running------*/
	if((mtx3x52_status = mtx3x52_runAcquisition (instrSession, VI_ON)) < 0)
		return mtx3x52_status;
        
	/*- Reads trace from the instrument--------------------------------------*/ 
	if((mtx3x52_status = mtx3x52_readChToArr (instrSession, channel, data, numberOfSamples, XStart, XIncrement)) < 0)
		return mtx3x52_status; 
                   
	return mtx3x52_status;
}
 
/*===========================================================================*/
/* Function: Configure MTX3x52 Vertical Settings                             */
/* Purpose:  This function configures the vertical settings for a specified  */
/*           channel                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_verticalSetup (ViSession instrSession, ViInt16 input,
					ViBoolean state, ViInt16 coupling, ViInt16 probe, 
					ViInt16 sense, ViReal64 offset, ViBoolean band)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(input, 1, 4))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViBooleanRange(state))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViInt16Range(coupling, 0, 2))
		return VI_ERROR_PARAMETER4;
	if (probe != 1 && probe != 10 && probe != 100 && probe != 1000)
		return VI_ERROR_PARAMETER5;
	if (mtx3x52_invalidViInt16Range(sense, 0, 17))
		return VI_ERROR_PARAMETER6;
    
	switch (probe) 
		{
		/*--Probe x1---------------------------------------------------------*/
		case 0 :	if (mtx3x52_invalidViInt16Range(sense,0,11))
						return VI_ERROR_MTX3x52_INCONSISTENT_PARAMETER6;
                    break;
        /*--Probe x10--------------------------------------------------------*/
        case 1 :    if (mtx3x52_invalidViInt16Range(sense,3,14))
                        return VI_ERROR_MTX3x52_INCONSISTENT_PARAMETER6; 
					break;
		/*--Probe x100-------------------------------------------------------*/ 
		case 2 :	if (mtx3x52_invalidViInt16Range(sense,6,17))
						return VI_ERROR_MTX3x52_INCONSISTENT_PARAMETER6; 
					break;
		/*--Probe x1000------------------------------------------------------*/
		case 3 :	if (mtx3x52_invalidViInt16Range(sense,9,20))
						return VI_ERROR_MTX3x52_INCONSISTENT_PARAMETER6; 
					break;  
		}      
	if (mtx3x52_invalidViReal64Range(offset,-cmdOffsetMax[sense],cmdOffsetMax[sense]))
		return VI_ERROR_PARAMETER7;
	if (mtx3x52_invalidViBooleanRange(band))
		return VI_ERROR_PARAMETER8;
    
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] 1\r", input);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 
	Fmt (wrBuffer, "DISP:TRAC:STAT%i[b2] %i[b2]\r", input, state);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 
	Fmt (wrBuffer, "INP%i[b2]:COUP %s\r", input, cmdVertCoupling[coupling]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 
	Fmt (wrBuffer, "SENS:BAND%i[b2]:AUTO %i[b2]\r", input, band);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:PTP %s\r", input, cmdVoltDiv[sense]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:OFFS %f\r", input, offset);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] %i[b2]\r", input, probe);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Configure MTX3x52 Horizontal Settings                           */
/* Purpose:  This function configures the horizontal settings of the         */
/*           instrument                                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_horizontalSetup (ViSession instrSession,
					ViInt16 range, ViReal64 pretrg) 
                  
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViReal64 convBuf;
	ViByte pretrgBuf[BUFFER_SIZE];
	ViInt32 retCntptr;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(range, 0, 34))
		return VI_ERROR_PARAMETER2;
      
	/*----------------------- time bases ------------------------------------*/
	Fmt (pretrgBuf, "DISP:TRAC:X:PDIV %s\r", cmdTimeDiv[range]);
	if ((mtx3x52_status = viWrite (instrSession, pretrgBuf, StringLength(pretrgBuf), &retCntptr)) < 0)
		return mtx3x52_status; 
        
	/*-Check input parameter ranges and test pretrig-------------------------*/    
	Fmt(&convBuf, "%f<%s", cmdTimeDiv[range]);
	if (mtx3x52_invalidViReal64Range(pretrg * cmdOffset[range],( -10 * convBuf), (25 * convBuf)))
		return VI_ERROR_PARAMETER3; 
  
	/*----------------  pretrig --------------------------------------------*/       
	Fmt (pretrgBuf, "TRIG:DEL %f\r", (pretrg * cmdOffset[range]));
	if ((mtx3x52_status = viWrite (instrSession, pretrgBuf, StringLength(pretrgBuf), &retCntptr)) < 0)
		return mtx3x52_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;

	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Configure MTX3x52 Horizontal Display                            */
/* Purpose:  This function configures the horizontal display                 */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_horizontalDisplay (ViSession instrSession, ViInt16 trace,
					ViInt16 x, ViInt16 y)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(trace,0,1))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range(x,1,2))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViInt16Range(y,1,2))
		return VI_ERROR_PARAMETER4;
   
	Fmt (wrBuffer,"DISP:TRAC:FORM %s;XY:XDEF INT%i[b2];YDEF INT%i[b2]\r",cmdDispTrace[trace],x,y);
	if ((mtx3x52_status = viWrite (instrSession,wrBuffer,StringLength(wrBuffer),&retCnt)) < 0)
		return mtx3x52_status;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
     
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: FFT                                                             */
/* Purpose:  This function configures the FFT                                */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_FFT (ViSession instrSession, ViBoolean fftvalid, ViInt16 calculate, 
						ViBoolean scale)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];     
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/

	if (mtx3x52_invalidViBooleanRange(fftvalid))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range(calculate,0,5))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViBooleanRange(scale))
		return VI_ERROR_PARAMETER4;
    
	/*-Set ON-OFF FFT, Calculate and scale parameters----------------*/ 
	Fmt (wrBuffer, "CALC:TRAN:FREQ %i[b2]\r", fftvalid);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Fmt (wrBuffer, "CALC:TRAN:FREQ:WIND %s\r", cmdcalcfft[calculate]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Fmt (wrBuffer, "DISP:TRAC:Y:SPAC %s\r", cmdScale[scale]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;   
    
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: General Trigger                                                 */
/* Purpose:  This function sets general parameters for trigger               */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_setGeneralTrigger (ViSession instrSession, ViInt16 channel,
										ViBoolean slope, ViInt16 coupling,
										ViBoolean noiseReject, ViInt16 type, ViReal64 level)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViReal64 convBuf;
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(channel,0,3))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViBooleanRange(slope))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViInt16Range(coupling,0,5))
		return VI_ERROR_PARAMETER4;
	if (mtx3x52_invalidViBooleanRange(noiseReject))
		return VI_ERROR_PARAMETER5;
	if (mtx3x52_invalidViInt16Range(type,0,2))
		return VI_ERROR_PARAMETER6;
	if (channel < 2)
		{
		if (mtx3x52_invalidViReal64Range(level,-800,800))
			return VI_ERROR_PARAMETER7;
		}
	else if (channel == 2)
		{ 
		if (mtx3x52_invalidViReal64Range(level,-800E-3,8E-3))
			return VI_ERROR_PARAMETER8;
    	
		}
    	
	/*-Set slope, coupling, level and filter parameters----------------*/
	Fmt (wrBuffer, "TRIG:SOUR %s\r", cmdSource[channel]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Fmt (wrBuffer, "TRIG:SLOP %s\r", cmdSlope[slope]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	if (channel < 3)
		{
		Fmt (wrBuffer, "TRIG:LEV %f\r", level);
		if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3x52_status;
		Fmt (wrBuffer, "TRIG:FILT:NOIS %i[b2]\r", noiseReject);
		if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3x52_status;
		switch (coupling)
			{
			case 0 :
					Fmt (wrBuffer, "TRIG:COUP %s\r", cmdTrigCoupling[coupling]);
					if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3x52_status;
					break;
			case 1 :
					Fmt (wrBuffer, "TRIG:COUP %s\r", cmdTrigCoupling[coupling]);
					if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3x52_status;
					break;	
			case 2 :
					Fmt (wrBuffer, "TRIG:FILT:HPAS 1\r");
					if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3x52_status;
					break;	
			case 3 :
					Fmt (wrBuffer, "TRIG:FILT:LPAS 1\r");
					if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3x52_status;
					break;	
			case 4 :
					Fmt (wrBuffer, "TRIG:VID:FIEL:SEL ALL\r");
					if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3x52_status;
					break;	
			case 5 :
					Fmt (wrBuffer, "TRIG:VID:LINE:SEL ALL\r");
					if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
						return mtx3x52_status;
					break;
			}
		}
    if(type<2)
    	{
		Fmt (wrBuffer, "TRIG:ATRIG %s\r", cmdType[type]);
		if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3x52_status;    
		}
	else 
		{
	
		Fmt (wrBuffer, "INIT:NAME EDGE\r");
		if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3x52_status; 
	
		}
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;   
    
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Display Options                                                 */
/* Purpose:  This function sets display options.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_setOptionDisplay (ViSession instrSession, ViBoolean grid,
					ViInt16 device_mode, ViBoolean avanced_mode, ViInt16 language, 
					ViInt16 standby, ViInt16 screen_saver, ViInt16 brightness, ViInt16 contrast)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViBooleanRange(grid))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range(device_mode,0,3))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViBooleanRange(avanced_mode))
		return VI_ERROR_PARAMETER4;
	if (mtx3x52_invalidViInt16Range(language,0,4))
		return VI_ERROR_PARAMETER5;
	if (mtx3x52_invalidViInt16Range(standby,0,3))
		return VI_ERROR_PARAMETER6;
	if (mtx3x52_invalidViInt16Range(screen_saver,0,3))
		return VI_ERROR_PARAMETER7;
	if (mtx3x52_invalidViInt16Range(brightness,0,100))
		return VI_ERROR_PARAMETER8;
	if (mtx3x52_invalidViInt16Range(contrast,0,100))
		return VI_ERROR_PARAMETER9;

	/*-Set option display parameters-----------------------------------------*/
	Fmt (wrBuffer, "DISP:TRAC:GRAT:GRID %i[b2]\r", grid);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Fmt (wrBuffer, "DEV:MOD %s\r", cmddevice[device_mode]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Delay (2.0);
	 
	Fmt (wrBuffer, "DEV:SCOP:EXP %i[b2]\r", avanced_mode);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Delay (2.0);
	        
	Fmt (wrBuffer, "SYST:LANG %s\r", cmdDisplayLang[language]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
	   return mtx3x52_status;      
	Delay (2.0);
	  
	Fmt (wrBuffer, "DISP:SCRE:SAVER %s\r", cmdDisplayScreen[screen_saver]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;   
	Fmt (wrBuffer, "SYST:POW:OFF %s\r", cmdSystemStandby[standby]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;   		
	Fmt (wrBuffer, "DISP:BRIG %f[p2]\r", brightness / 100.0);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Fmt (wrBuffer, "DISP:CONT %f[p2]\r", contrast / 100.0);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status; 		

	/*-Check instrument status & Return from function------------------------*/				   
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
      
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Display Waveform                                                */
/* Purpose:  This function sets waveform display.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_setWaveformDisplay (ViSession instrSession,ViInt16 acquisition,
					ViInt16 average_rate, ViInt16 mode, ViBoolean repet)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(acquisition, 0,1))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range(average_rate, 0, 4))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViInt16Range(mode, 0, 1))
		return VI_ERROR_PARAMETER4;
	if (mtx3x52_invalidViBooleanRange(repet))
		return VI_ERROR_PARAMETER5;

	/*-Set waveform display parameters---------------------------------------*/
	Fmt (wrBuffer, "AVER:TYPE %s\r", cmdMaxminAcq[acquisition]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Fmt (wrBuffer, "AVER %i[b2]\r", repet);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Fmt (wrBuffer, "DISP:TRACE:MODE %s\r", cmdMode[mode]);
		if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3x52_status;
	if(repet==1)
		{
		Fmt (wrBuffer, "AVER:COUN %s\r", cmdAverage[average_rate]);
		if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
			return mtx3x52_status;
    	}
	
    /*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Set Cursors                                                     */
/* Purpose:  This function sets MTX3x52 cursors.                              */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_setCursors (ViSession instrSession, ViInt16 cursor,
					ViInt16 reference, ViBoolean unattcur, ViReal64 t1, ViReal64 t2, ViReal64 tphase)
{                              
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(cursor,0,2))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range(reference,0,1))
		return VI_ERROR_PARAMETER3;
	if ((mtx3x52_invalidViBooleanRange(unattcur))) 
		return VI_ERROR_PARAMETER4;
	if (mtx3x52_invalidViReal64Range(t1,0,2000.0))
		return VI_ERROR_PARAMETER5;
	if (mtx3x52_invalidViReal64Range(t2,0,2000.0))
		return VI_ERROR_PARAMETER6;
	if (mtx3x52_invalidViReal64Range(tphase,0,2000))
		return VI_ERROR_PARAMETER7;    
	 
	/*-Set cursors-----------------------------------------------------------*/    
	Fmt (wrBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
    
	switch(cursor)
		{
		case 0 :
			Fmt (wrBuffer, "DISP:CURS:STAT OFF;PHASE:STAT OFF\r");
			if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
				return mtx3x52_status;
			break;
			
		case 1 :
			Fmt (wrBuffer, "DISP:CURS:STAT ON;PHASE:STAT OFF\r");
			if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
				return mtx3x52_status;
			if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS %lf\r",t1)) < 0)
				return mtx3x52_status;
			Delay (1.7);
            	
			if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS %lf\r",t2)) < 0)
				return mtx3x52_status;
			break;
		
		case 2 :
			Fmt (wrBuffer, "DISP:CURS:STAT OFF;PHASE:STAT ON\r");
			if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
				return mtx3x52_status;
			if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS %lf\r",t1)) < 0)
				return mtx3x52_status;
			Delay (1.7);	
            	
			if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS %lf\r",t2)) < 0)
				return mtx3x52_status;
			if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:TIME3:POS %lf\r",tphase)) < 0)
				return mtx3x52_status;	
			break;	
    	
		}
	Fmt (wrBuffer, "DISP:CURS:AUTO:STAT %i[b2]\r", unattcur);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Hardcopy                                                        */
/* Purpose:  This function configures MTX3x52 hardcopy facility.              */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_setHardcopy (ViSession instrSession, ViInt16 device,
					ViInt16 port, ViString filename, ViBoolean type)
{                              
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
            
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(device,0,7))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range(port,0,3))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViBooleanRange(type))
		return VI_ERROR_PARAMETER5;
	    
	/*-Set hardcopy parameters-----------------------------------------------*/
	if (port == 3)
		{
		if (StringLength(filename) != 8) 
			return VI_ERROR_PARAMETER4;
		if ((mtx3x52_status = viPrintf (instrSession,"HCOP:DEST \"%s\"\r",filename)) < 0)
			return mtx3x52_status;
		if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
		}	
	Fmt (wrBuffer, "HCOP:DEST \"%s\"\r",cmdDestination[port]);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	Fmt (wrBuffer, "HCOP:DEV:LANG %s; COL %i[b2]\r", cmdHardcopyLang[device], type);
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Run Autoset                                                     */
/* Purpose:  This function performs an automatic setup of the MTX3x52.       */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_runAutoset (ViSession instrSession)
{                              
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViInt16 i,state;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	Fmt (wrBuffer, "AUTOS:EXE\r");
	if ((mtx3x52_status = viWrite (instrSession, wrBuffer, StringLength(wrBuffer), &retCnt)) < 0)
		return mtx3x52_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Run Acquisition                                                 */
/* Purpose:  This function performs an acquisition session.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_runAcquisition (ViSession instrSession, ViBoolean run)
{                              
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViInt32 retCnt;
	ViByte rdBuffer[BUFFER_SIZE];
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViBooleanRange(run))
		return VI_ERROR_PARAMETER3;
    
	/*-Get current trigger mode----------------------------------------------*/
	Fmt (rdBuffer, "TRIG:RUN:STAT %i[b2]\r", run);
	if ((mtx3x52_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx3x52_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
    return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Run Hardcopy                                                    */
/* Purpose:  This function performs a hardcopy.                              */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_runHardcopy (ViSession instrSession, ViBoolean action)
{                              
	ViStatus mtx3x52_status = VI_SUCCESS;
    
	if (action) 
		{               
		if ((mtx3x52_status = viPrintf (instrSession,"HCOP:SDUM\r")) < 0)
			return mtx3x52_status;
		}
	else {
		if ((mtx3x52_status = viPrintf (instrSession,"HCOP:SDUM:UNDO\r")) < 0)
			return mtx3x52_status;
	}   
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Read Cursors                                                    */
/* Purpose:  This function reads MTX3x52 cursors.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_readCursors (ViSession instrSession, ViInt16 reference,
					ViPReal64 t1, ViPReal64 t2, ViPReal64 dt, ViPReal64 tphase, ViPReal64 v1, 
					ViPReal64 v2, ViPReal64 dv, ViPReal64 vphase)
{                              
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViInt16 valtphase;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(reference,0,1))
		return VI_ERROR_PARAMETER2;

	/*-Display cursor menu---------------------------------------------------*/
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mtx3x52_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx3x52_status;
        
	/*-Set cursor positions--------------------------------------------------*/
	if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS?\r")) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viScanf (instrSession,"%lf",t1)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS?\r")) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viScanf (instrSession,"%lf",t2)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:PHAS:STAT?\r")) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viScanf (instrSession, "%i", &valtphase)) < 0)
		return mtx3x52_status;
	if(valtphase == 1)	
		{
		if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:TIME3:POS?\r")) < 0)
			return mtx3x52_status;
		if ((mtx3x52_status = viScanf (instrSession,"%lf",tphase)) < 0)
			return mtx3x52_status;
		if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:VOLT3:POS?\r")) < 0)
			return mtx3x52_status;
		if ((mtx3x52_status = viScanf (instrSession,"%lf",vphase)) < 0)
			return mtx3x52_status;
		}
	if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS1:VOLT:POS?\r")) < 0)
		return mtx3x52_status;

	if ((mtx3x52_status = viScanf (instrSession,"%lf",v1)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viPrintf (instrSession,"DISP:CURS:VOLT2:POS?\r")) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viScanf (instrSession,"%lf",v2)) < 0)
		return mtx3x52_status;
		
	/*-Calculate 'delta t' & 'delta v'---------------------------------------*/
	*dt = *t2 - *t1;
	*dv = *v2 - *v1;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Read Measurement                                                */
/* Purpose:  This function reads a specified measurement.                    */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_readMeasurement (ViSession instrSession, ViInt16 measure,
					ViInt16 reference, ViPReal64 reading)
{                              
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(measure,0,18))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range(reference,0,1))
		return VI_ERROR_PARAMETER3;
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mtx3x52_status = viWrite (instrSession, rdBuffer, StringLength(rdBuffer), &retCnt)) < 0)
		return mtx3x52_status;
	if(measure<18) 
		{
		if ((mtx3x52_status = viPrintf (instrSession,"MEAS:%s? %s\r",cmdReadMeasure[measure],cmdRef[reference])) < 0)
			return mtx3x52_status;
		}
	else
		{
		if ((mtx3x52_status = viPrintf (instrSession,"MEAS:PHAS? %s,%s\r", cmdRef[reference] ,cmdRef[!reference])) < 0)
			return mtx3x52_status;
		}	
	if ((mtx3x52_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = Scan (rdBuffer,"%s>%f",reading))!=1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
     
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Read Channel To Array                                           */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_readChToArr (ViSession instrSession, ViInt16 source,			   
					ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{   
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,size, index;
	ViInt16 tstRes;
	ViUInt16 interface;
	static ViByte tabByte[MAX_MEM_SIZE];
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(source,0,1))
		return VI_ERROR_PARAMETER2;
    
	/*-Check if source is active---------------------------------------------*/
	if ((mtx3x52_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", source + 1)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mtx3x52_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_MTX3x52_CHAN_NOT_ACTIVE;
    
	/*-Modify Time Out for big waveform transfers------------------------*/
	if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
		return mtx3x52_status;

	/*-Modify Protocol for binary file transfers-------------------------*/
	viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
	if (interface == VI_INTF_ASRL)
	{
		if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
			return mtx3x52_status;
	}

	/*-Set data format and start transfert-----------------------------------*/    
	if ((mtx3x52_status = viPrintf (instrSession,"TRAC? INT%hd\r", source + 1)) < 0)
		return mtx3x52_status;
	if (interface==VI_INTF_ASRL) 
	{
		/* Read 11 characters (#an header) to define the length of the binary block following */
		if ((mtx3x52_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
			return mtx3x52_status;
		index = FindPattern (tabByte, 0, HEADER_SIZE, "#", 0, 0);
		headerSize = tabByte[index + 1] - 0x30;
		Scan(tabByte + index + 2, "%i[b4]", &size);
		if ((mtx3x52_status = viRead (instrSession, tabByte + HEADER_SIZE, index + size + 7 + headerSize - HEADER_SIZE, &retCnt)) < 0)
			return mtx3x52_status;
		/*-Reset Protocol for binary file transfers----------------------------*/
		if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
			return mtx3x52_status;
	}
	else if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
		return mtx3x52_status;

	/*-Reset Time Out ---------------------------------------------------*/
	if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return mtx3x52_status;
		
	/*-Make data readable----------------------------------------------------*/
	if ((mtx3x52_status = mtx3x52_makeDataReadable(tabByte,waveformArray,numberOfSamples,XStart,XIncrement)) < 0)
		return mtx3x52_status;
                            
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Read Channel To File                                            */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in a file                                        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_readChToFile ( ViSession instrSession, ViInt16 channel,
                    ViInt16 drive, ViString filename, ViBoolean overwrite)
{                        
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 fileSize, retCnt, size, index, RunState;													
	static ViByte tabByte[MAX_MEM_SIZE];
    ViInt16 tstRes;
    ViString filenameTemp = "___tmp__.trc";

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(channel,0,1))
		return VI_ERROR_PARAMETER2;								 
	if (mtx3x52_invalidViInt16Range(drive,0,1))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER5;
        
	/*-Check if source is active---------------------------------------------*/
	if ((mtx3x52_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", channel + 1)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mtx3x52_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_MTX3x52_CHAN_NOT_ACTIVE;
    
	/*-Check filename--------------------------------------------------------*/
	if (mtx3x52_invalidFilename (filename,drive,".TRC"))
		return VI_ERROR_PARAMETER4;

	/* if oscilloscope is in Run mode ?*/
	viWrite (instrSession, "TRIG:RUN:STAT?\r", 15, &RunState);
	viScanf (instrSession, "%hd", &RunState);
	if (RunState)
		viWrite (instrSession, "TRIG:RUN:STAT 0\r", 16, &RunState);
	
	/*-Store current trace into MTX3x52---------------------------------------*/
	if (drive) 
		{
		/*-Check overwrite---------------------------------------------------*/
		if ((mtx3x52_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
			return mtx3x52_status;
		if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3x52_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
			{
			if (overwrite) 
				{
				if ((mtx3x52_status = viPrintf (instrSession,"MMEM:DEL \"%s\"\r", filename)) < 0)
					return mtx3x52_status;
				}
			else 
				return VI_ERROR_MTX3x52_EXISTING_FILE;
			}
        
		/*-Perform transfer--------------------------------------------------*/
		if ((mtx3x52_status = viPrintf (instrSession,"MMEM:STOR:TRAC INT%hd,\"%s\"\r",
					channel+1,filename )) < 0)
			return mtx3x52_status;
		}
    
	/*-Store current trace into a PC file------------------------------------*/
	else 
		{
		/*-Check overwrite---------------------------------------------------*/
		if (GetFileInfo(filename,&fileSize)  && !overwrite) 
			return VI_ERROR_MTX3x52_EXISTING_FILE;

		if (GetFileInfo (filename,&fileSize)  && overwrite) 
			DeleteFile (filename);

		/*-Perform transfer--------------------------------------------------*/
		if ((mtx3x52_status = viPrintf (instrSession,"MMEM:STOR:TRAC INT%hd,\"%s\"\r",
					channel+1, filenameTemp )) < 0)
			return mtx3x52_status;

		/*-Temp file transfer------------------------------------------------*/
		if ((mtx3x52_status = mtx3x52_fileTransfer (instrSession, filenameTemp, 1, VI_ON)) < 0)
			return mtx3x52_status;

		/*-Rename Temp file--------------------------------------------------*/
		RenameFile (filenameTemp, filename);
			
		/*-Delete Temp file--------------------------------------------------*/
		if ((mtx3x52_status = viPrintf (instrSession,"MMEM:DEL \"%s\"\r", filenameTemp)) < 0)
			return mtx3x52_status;
		}

	/*Restart the oscilloscope in Run mode*/
	if (RunState)
		viWrite (instrSession, "TRIG:RUN:STAT 1\r", 16, &RunState);
  
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Read File To Array                                              */
/* Purpose:  This function reads a file from the PC or from the instrument   */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_readFileToArr (ViSession instrSession, ViInt16 drive,
					ViString filename, ViReal64 _VI_FAR waveformArray[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{   
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,fileSize, size, index;
	static ViByte tabByte[MAX_MEM_SIZE];
	ViUInt16 interface;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(drive,0,1))
		return VI_ERROR_PARAMETER2;
	        
	/*-Check filename--------------------------------------------------------*/
	if (mtx3x52_invalidFilename (filename,drive,".TRC"))
		return VI_ERROR_PARAMETER3;
        
	/*-Read file from MTX3x52-------------------------------------------------*/
	if (drive) 
		{
		/*-Check filename----------------------------------------------------*/
		if ((mtx3x52_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
			return mtx3x52_status;
		if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3x52_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
			return VI_ERROR_MTX3x52_FILE_NOT_FOUND;
        
		/*-Modify Time Out for big waveform transfers------------------------*/
		if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
			return mtx3x52_status;
	
		/*-Modify Protocol for binary file transfers-------------------------*/
		viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
		if(interface==VI_INTF_ASRL)
			{
			if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
				return mtx3x52_status;
			}

		/*-Perform transfer--------------------------------------------------*/
		if ((mtx3x52_status = viPrintf (instrSession,"MMEM:DATA? \"%s\"\r",filename)) < 0)
			return mtx3x52_status;
		if (interface==VI_INTF_ASRL) 
			{
			/* Read 11 characters (#an header) to define the length of the binary block following */
			if ((mtx3x52_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
				return mtx3x52_status;
			index = FindPattern (tabByte, 0, -1, "#", 0, 0);
			headerSize = tabByte[index + 1] - 0x30;
			Scan(tabByte + index + 2, "%i[b4]", &size);
			if ((mtx3x52_status = viRead (instrSession, tabByte + HEADER_SIZE, size + 4 + headerSize - HEADER_SIZE, &retCnt)) < 0)
				return mtx3x52_status;
				
			retCnt += HEADER_SIZE;

			/*-Reset Protocol for binary file transfers----------------------------*/
			if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				return mtx3x52_status;
			}
				
		/*-Reset Time Out ---------------------------------------------------*/
		if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,TMO_VALUE)) < 0)
			return mtx3x52_status;
    	}
	/*-Read file from PC-----------------------------------------------------*/
	else 
		{
		/*-Check filename----------------------------------------------------*/
		if (!GetFileInfo(filename,&fileSize)) 
			return VI_ERROR_MTX3x52_FILE_NOT_FOUND;
        if ((FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, MAX_MEM_SIZE,MAX_MEM_SIZE,
			VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY)) < 0)
		return VI_ERROR_INSTR_FILE_OPEN;
		}
		
	/*-Make data readable----------------------------------------------------*/
	if ((mtx3x52_status = mtx3x52_makeDataReadable(tabByte,waveformArray,numberOfSamples, XStart, XIncrement)) < 0)
		return mtx3x52_status;
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
        
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Write File To Instrument Display Memory                         */
/* Purpose:  This function fills the instrument display memory.              */
/*           Data are in the specified file.                                 */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_writeFileToMem(ViSession instrSession, ViString filename, ViInt16 memory)
{   
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViUInt32 retCnt;
	static ViByte tabByte[MAX_MEM_SIZE];

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(memory,1,4))
		return VI_ERROR_PARAMETER4;

	/*-Check filename--------------------------------------------------------*/
	if (mtx3x52_invalidFilename (filename,1,".TRC"))
		return VI_ERROR_PARAMETER3;
    
	/*-Check filename----------------------------------------------------*/
	if ((mtx3x52_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
		return mtx3x52_status;
	if (FindPattern (tabByte, 0, -1, filename , 0, 0) < 0)
		return VI_ERROR_MTX3x52_FILE_NOT_FOUND;

	/*-Perform transfer--------------------------------------------------*/
	if ((mtx3x52_status = viPrintf (instrSession,"MMEM:LOAD:TRAC TRACE%hd,\"%s\"\r",
	            memory, filename)) < 0)
		return mtx3x52_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
    
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Save/Recall Setup From/To Instrument                            */
/* Purpose:  This function allows you to save the current setup of the       */
/*           instrument or to recall a setup stored in a file.               */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_saveSetup (ViSession instrSession, ViBoolean mode, ViInt16 drive,
				ViString filename, ViBoolean overwrite)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViUInt32 retCnt, fileSize;
	static ViByte string[2000],tabByte[MAX_MEM_SIZE];
	ViUInt16 interface;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViBooleanRange(mode))
		return VI_ERROR_PARAMETER2;
	if (mtx3x52_invalidViInt16Range(drive,0,1))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER5;

	/*-Check filename--------------------------------------------------------*/
	if (mtx3x52_invalidFilename (filename,drive,".CFG"))
		return VI_ERROR_PARAMETER4;

	/*-Save setup from MTX3x52------------------------------------------------*/
	if (mode) 	
		{
		/*-Target drive is MTX3x52--------------------------------------------*/
		if (drive) 
			{
			/*-Check overwrite-----------------------------------------------*/
			if ((mtx3x52_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
				return mtx3x52_status;
			if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
				return mtx3x52_status;
			if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
				{
				if (overwrite) 
					{
					if ((mtx3x52_status = viPrintf (instrSession, "MMEM:DEL \"%s\"\r", filename)) < 0)
						return mtx3x52_status;
					}
                else return VI_ERROR_MTX3x52_EXISTING_FILE;
				}
			/*-Perform transfer----------------------------------------------*/
			if ((mtx3x52_status = viPrintf (instrSession, "MMEM:STORE:STATE \"%s\"\r",filename)) < 0)
				return mtx3x52_status;
			}
		/*-Target drive is PC------------------------------------------------*/
		else 
			{
			/*-Check overwrite-----------------------------------------------*/    
			if (GetFileInfo(filename,&fileSize)  && !overwrite) 
				return VI_ERROR_MTX3x52_EXISTING_FILE;
            
			/*-Modify Protocol for binary file transfers---------------------*/
			viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
			if(interface==VI_INTF_ASRL)
				{
				if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
					return mtx3x52_status;
				}
			/*-Perform transfer----------------------------------------------*/    
			if ((mtx3x52_status = viPrintf (instrSession,"SYST:SET?\r")) < 0)
				return mtx3x52_status;
            
			if (interface==VI_INTF_ASRL) 
				{
				/* Read 11 characters (#an header) to define the length of the binary block following */
				if ((mtx3x52_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
				    return mtx3x52_status;

				headerSize = tabByte[1] - 0x30;
				Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
				if ((mtx3x52_status = viRead (instrSession, tabByte + HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
				     return mtx3x52_status;

				retCnt += HEADER_SIZE;

				/*-Reset Protocol for binary file transfers----------------------------*/
				if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				    return mtx3x52_status;
				}
			else if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
				return mtx3x52_status;
            
			/*-Fill PC file--------------------------------------------------*/    
			if ((ArrayToFile (filename, tabByte, VAL_UNSIGNED_CHAR, retCnt - 2,retCnt - 2, VAL_GROUPS_TOGETHER,
					VAL_GROUPS_AS_ROWS,VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0) 
				return VI_ERROR_INSTR_FILE_WRITE;
			}
		}
    
	/*-Recall setup to MTX3x52------------------------------------------------*/
	else 
		{
		/*-Source drive is MTX3x52--------------------------------------------*/
		if (drive) 
			{
			/*-Check file name-----------------------------------------------*/    
			if ((mtx3x52_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
				return mtx3x52_status;
			if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE,&retCnt)) < 0)
				return mtx3x52_status;
			if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
				return VI_ERROR_MTX3x52_FILE_NOT_FOUND;
				
			/*-Perform transfer----------------------------------------------*/
			if ((mtx3x52_status = viPrintf (instrSession,"MMEM:LOAD:STATE \"%s\"\r",filename)) < 0)
				return mtx3x52_status;
			}
			
		/*-Source drive is PC------------------------------------------------*/
		else 
			{
			/*-Check file name-----------------------------------------------*/
			if (!GetFileInfo(filename,&fileSize)) 
				return VI_ERROR_MTX3x52_FILE_NOT_FOUND;
			/*-Read PC file--------------------------------------------------*/
			if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, 20, 20,VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
				return VI_ERROR_INSTR_FILE_OPEN;
			Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
			headerSize = 6;
			if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, headerSize + fileSize, 
					headerSize + fileSize, VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
				return VI_ERROR_INSTR_FILE_OPEN;
			/*-Perform transfer----------------------------------------------*/
			Fmt(string,"%s<SYST:SET %s[t-w*q]\r", headerSize + fileSize, tabByte);
			if ((mtx3x52_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
				return mtx3x52_status;
			}																  
		Delay(0.5);	
		}
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;
	 
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: File Transfer                                                   */ 
/* Purpose:  This function allows you to transfer a file from a drive        */
/*           to another.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_fileTransfer (ViSession instrSession, ViString filename,
							ViInt16 srcDrive, ViBoolean overwrite)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte trueFilename[MAX_FILENAME_LEN];
	ViInt16 headerSize;
	ViInt32 retCnt,fileSize, RunState;
	static ViByte tabByte[MAX_MEM_SIZE],tempString[20];
	ViUInt16 interface;

	/*-Check input parameter ranges------------------------------------------*/
	if (mtx3x52_invalidViInt16Range(srcDrive,0,1))
		return VI_ERROR_PARAMETER3;
	if (mtx3x52_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER4;
  
	/*-Check filename--------------------------------------------------------*/
	if ((FindPattern (filename, 0, -1, ".", 0, 0) > 8) && (StringLength(filename) > 12))
		return VI_ERROR_PARAMETER2;

	/*-Computer to MTX3x52 transfer-------------------------------------------*/
	if (!srcDrive ) 
		{
		/*-Check file name---------------------------------------------------*/    
		if (!GetFileInfo(filename,&fileSize)) 
			return VI_ERROR_MTX3x52_FILE_NOT_FOUND;
		SplitPath (filename, NULL, NULL, trueFilename);

		/* if oscilloscope is in Run mode ?*/
		viWrite (instrSession, "TRIG:RUN:STAT?\r", 15, &RunState);
		viScanf (instrSession, "%hd", &RunState);
		if (RunState)
			viWrite (instrSession, "TRIG:RUN:STAT 0\r", 16, &RunState);

		/*-Check overwrite---------------------------------------------------*/    
		if ((mtx3x52_status = viPrintf (instrSession, "MMEM:CAT?\r")) < 0)
			return mtx3x52_status;
		if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3x52_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) != -1) 
			{
			if (overwrite) 
				{
				if ((mtx3x52_status = viPrintf (instrSession, "MMEM:DEL \"%s\"\r", filename)) < 0)
					return mtx3x52_status;
				}
			else   
				return VI_ERROR_MTX3x52_EXISTING_FILE;
			}
    
		/*-Read PC file------------------------------------------------------*/
		GetFileInfo (filename, &fileSize);
     
		/*-Read any file but not ".TRC" or ".CFG"----------------------------*/
		if ((FindPattern(trueFilename, 0, -1, "TRC", 0, 0)==-1) && (FindPattern(trueFilename, 0, -1, "CFG", 0, 0)==-1)) 
			{
			/*-Perform transfer----------------------------------------------*/
			Fmt (tempString, "%i[b4]", fileSize);
			Fmt (tabByte, "MMEM:DATA \"%s\", #%i[b2]%i[b4]", trueFilename, StringLength(tempString), fileSize);
			}
        
		/*-Read ".TRC" or ".CFG" files---------------------------------------*/
		else 
			{
			/*-Perform transfer--------------------------------------------------*/
			Fmt (tabByte, "MMEM:DATA \"%s\",", trueFilename);
			}
			
		headerSize = StringLength(tabByte);

		if (FileToArray (filename, tabByte + headerSize, VAL_UNSIGNED_CHAR, fileSize, 1,
		                	VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
			return VI_ERROR_INSTR_FILE_OPEN;

		tabByte[headerSize + fileSize] = '\r';
		
		if ((mtx3x52_status = viWrite (instrSession, tabByte, headerSize + fileSize + 1, &retCnt)) < 0) 
			return mtx3x52_status;
		Delay(1.5);	
		
		/*Restart the oscilloscope in Run mode*/
		if (RunState)
			viWrite (instrSession, "TRIG:RUN:STAT 1\r", 16, &RunState);
		}
    
	/*-MTX3x52 to Computer transfer-------------------------------------------*/
	else 
		{
		/*-Check filename----------------------------------------------------*/
		if ((mtx3x52_status = viPrintf (instrSession, "MMEM:CAT?\r")) < 0)
			return mtx3x52_status;
		if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3x52_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
			return VI_ERROR_MTX3x52_FILE_NOT_FOUND;

		/*-Check overwrite---------------------------------------------------*/    
		if (GetFileInfo(filename, &fileSize)  && !overwrite) 
			return VI_ERROR_MTX3x52_EXISTING_FILE;

		/*-Modify Time Out for big file transfers----------------------------*/
		if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
			return mtx3x52_status;

		/*-Modify Protocol for binary file transfers-------------------------*/
		viGetAttribute (instrSession, VI_ATTR_INTF_TYPE, &interface);
		if(interface==VI_INTF_ASRL)
			{
			if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
				return mtx3x52_status;
			}

		/*-Perform transfer--------------------------------------------------*/
		if ((mtx3x52_status = viPrintf (instrSession, "MMEM:DATA? \"%s\"\r", filename)) < 0)
			return mtx3x52_status;

		if (interface==VI_INTF_ASRL) 
        	{
			/* Read 11 characters (#an header) to define the length of the binary block following */
			if ((mtx3x52_status = viRead (instrSession, tabByte, HEADER_FILE, &retCnt)) < 0)
				return mtx3x52_status;

			headerSize = tabByte[1] - 0x30;
			Scan(tabByte, "%s[i2]>%i[b4]", &fileSize);

			if ((mtx3x52_status = viRead (instrSession, tabByte+HEADER_FILE, fileSize+4+headerSize-HEADER_FILE, &retCnt)) < 0)
				return mtx3x52_status;

			retCnt += HEADER_FILE;

			/*-Reset Protocol for binary file transfers----------------------------*/
			if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
				return mtx3x52_status;
			}
		else if ((mtx3x52_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mtx3x52_status;

		/*-Reset Time Out ---------------------------------------------------*/
		if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
			return mtx3x52_status;

		/*-Parse non ".TRC" or ".CFG" files----------------------------------*/
		if ((FindPattern(filename,0,-1,"TRC",0,0)==-1) && (FindPattern(filename, 0, -1, "CFG", 0, 0)==-1))
			headerSize = tabByte[1] - 0x30 + 2; 
		else
			headerSize = 0;

		/*-Fill PC file------------------------------------------------------*/    
		if ((ArrayToFile (filename, tabByte + headerSize, VAL_UNSIGNED_CHAR, retCnt - headerSize - 2, 1, 
		VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_CONST_WIDTH, 4, VAL_BINARY, VAL_TRUNCATE)) < 0) 
			return VI_ERROR_INSTR_FILE_WRITE;
		}

	/*-Check instrument status & Return from function------------------------*/
	if ((mtx3x52_status = mtx3x52_instrStatus(instrSession)) < 0)
		return mtx3x52_status;

	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
	ViStatus mtx3x52_status = VI_SUCCESS;

	if ((mtx3x52_status = viPrintf (instrSession, "%s", writeBuffer)) < 0)
		return mtx3x52_status;

	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_readInstrData (ViSession instrSession, ViInt32 numBytes,
							ViChar _VI_FAR rdBuf[], ViPInt32 bytesRead)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	*bytesRead = 0L;

	if ((mtx3x52_status = viRead (instrSession, rdBuf, BUFFER_SIZE, bytesRead)) < 0)
		return mtx3x52_status;

	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx3x52_reset (ViSession instrSession)
{
	ViUInt32 retCnt = 0;
	ViStatus  mtx3x52_status = VI_SUCCESS;

	/*-Initialize the instrument to a known state----------------------------*/
	if (( mtx3x52_status = viWrite (instrSession, "\r*RST\r", 6, &retCnt)) < 0)
		return  mtx3x52_status;

	if (( mtx3x52_status =  mtx3x52_defaultInstrSetup (instrSession)) < 0)  
		return  mtx3x52_status;
        
	return  mtx3x52_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_selfTest (ViSession instrSession, ViPInt16 testResult,
								ViChar _VI_FAR testMessage[])
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,20 * TMO_VALUE)) < 0)
		return mtx3x52_status; 
    
	if ((mtx3x52_status = viWrite (instrSession, "*TST?\r", 6, &retCnt)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return mtx3x52_status;
	if (Scan (rdBuffer, "%i[b1]", testResult) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	if (*testResult) 
		Fmt(testMessage, "%s<Self test failed");
	else 
		Fmt(testMessage, "%s<No Error");  

	if ((mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 
		TMO_VALUE)) < 0)
		return mtx3x52_status;

	return mtx3x52_status;
}


/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_errorQuery (ViSession instrSession, ViPInt32 errCode,
							ViChar _VI_FAR errMessage[])
{
	ViUInt32 retCnt = 0;
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx3x52_status = viWrite (instrSession, "SYST:ERR?\r", 10, &retCnt)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mtx3x52_status;

	if (Scan (rdBuffer, "%i[b4],%s[t10]", errCode,errMessage) != 2)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_errorMessage (ViSession instrSession, ViStatus errorCode,
							ViChar _VI_FAR errMessage[])
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViInt16 i;
    
	static mtx3x52_tStringValPair statusDescArray[] = {
		{VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
		{VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
		{VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
		{VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
		{VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
		{VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
		{VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
		{VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
		{VI_ERROR_INSTR_PARAMETER9,   "ERROR: Parameter 9 out of range"},
		{VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
		{VI_ERROR_INSTR_FILE_OPEN,    "ERROR: Opening the specified file"},
		{VI_ERROR_INSTR_FILE_WRITE,   "ERROR: Writing to the specified file"},
		{VI_ERROR_INSTR_INTERPRETING_RESPONSE,           "ERROR: Interpreting the instrument's response"},
		{VI_ERROR_MTX3x52_ERROR,                  "ERROR: Instrument specific error"},
		{VI_ERROR_MTX3x52_INCONSISTENT_PARAMETER3,"ERROR: Parameter 3 is INCONSISTENT"},
		{VI_ERROR_MTX3x52_INCONSISTENT_PARAMETER5,"ERROR: Parameter 5 is INCONSISTENT"},
		{VI_ERROR_MTX3x52_INCONSISTENT_PARAMETER6,"ERROR: Parameter 6 is INCONSISTENT"},
		{VI_ERROR_MTX3x52_INCONSISTENT_PARAMETER7,"ERROR: Parameter 7 is INCONSISTENT"},
		{VI_ERROR_MTX3x52_AUTOTEST_PROBLEM,       "ERROR: Performing a self-test"}, 
		{VI_ERROR_MTX3x52_CHAN_NOT_ACTIVE,        "ERROR: Channel is not active"},
		{VI_ERROR_MTX3x52_NO_CHAN_ENABLED,        "ERROR: No channel is enabled"},
		{VI_ERROR_MTX3x52_FILE_NOT_FOUND,         "ERROR: File not available on support"},
		{VI_ERROR_MTX3x52_EXISTING_FILE,          "ERROR: Filename already used"},
		{VI_ERROR_MTX3x52_INVALID_FILE_CONTENT,   "ERROR: File content is invalid"},
		{VI_NULL, VI_NULL}  };
	          
	mtx3x52_status = viStatusDesc (instrSession, errorCode, errMessage);
	if (mtx3x52_status == VI_WARN_UNKNOWN_STATUS) 
		{
		for (i=0; statusDescArray[i].stringName; i++) 
			{
			if (statusDescArray[i].stringVal == errorCode) 
				{
				Fmt (errMessage, "%s<%s", statusDescArray[i].stringName);
				return (VI_SUCCESS);
				}
			}
		Fmt (errMessage, "%s<Unknown Error 0x%x[uw8p0]", errorCode);
		return (VI_WARN_UNKNOWN_STATUS);
		}
	mtx3x52_status = VI_SUCCESS;
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC mtx3x52_revisionQuery (ViSession instrSession,
					ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
	ViUInt32 retCnt = 0;
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mtx3x52_status = viWrite (instrSession, "*IDN?\r", 6, &retCnt)) < 0)
		return mtx3x52_status;
	if ((mtx3x52_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mtx3x52_status;
	if (Scan (rdBuffer, "%s[t10]", instrRev) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
	Fmt (driverRev, "%s<%s", MTX3x52_REVISION);

	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mtx3x52_close (ViSession instrSession)
{

	mtx3x52_instrRange instrPtr;
	ViSession rmSession;
	ViStatus  mtx3x52_status = VI_SUCCESS;

	if (( mtx3x52_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return  mtx3x52_status;
	if (( mtx3x52_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return  mtx3x52_status;
    
	if(instrPtr != NULL)
		free (instrPtr);
    
	mtx3x52_status = viClose (instrSession);
	viClose (rmSession);

	return  mtx3x52_status;
}

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean  mtx3x52_invalidViBooleanRange (ViBoolean val)
{
	return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx3x52_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx3x52_invalidViInt32Range  (ViInt32 val, ViInt32 min, ViInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx3x52_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mtx3x52_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mtx3x52_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mtx3x52_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Instrument Status                                               */
/* Purpose:  This function checks the instrument error status, by reading    */
/*           and processing instrument's event status register. An error     */
/*           is reported if bits 4 and 5 resp. EXE and CME are asserted.     */
/*===========================================================================*/
ViStatus  mtx3x52_instrStatus (ViSession instrSession)
{																							 
	ViUInt32 retCnt = 0;
	ViStatus  mtx3x52_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte status;
    
	/*- Command to query Event Status Register ------------------------------*/
	if (( mtx3x52_status = viPrintf (instrSession,"*ESR?\r")) < 0)
		return  mtx3x52_status;
        
	if (( mtx3x52_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return  mtx3x52_status;

	/*- Scanning the response -----------------------------------------------*/
	if (Scan (rdBuffer, "%i[b1]", &status) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
	/*- Testing bits 4 and 5 ------------------------------------------------*/
	if (status & 0x30) 
		return VI_ERROR_MTX3x52_ERROR;
    
	return  mtx3x52_status;
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the  mtx3x52_init function.  When */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from  mtx3x52_init to     */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus  mtx3x52_initCleanUp (ViSession openRMSession,ViPSession openInstrSession,
                    ViStatus currentStatus)
{
	viClose (*openInstrSession);
	viClose (openRMSession);
	*openInstrSession = VI_NULL;
    
	return currentStatus;
}

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the  mtx3x52_reset operation and by the   */
/*            mtx3x52_init function if the reset option has not been         */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus  mtx3x52_defaultInstrSetup (ViSession instrSession)
{
	ViStatus  mtx3x52_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	mtx3x52_instrRange instrPtr;
        
	/* Determine if the structure has been initialized for the current VISA  */
	/* Session and malloc if it has not.                                     */
	if ( mtx3x52_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
		return  mtx3x52_status;
    
	if (instrPtr == NULL) 
		instrPtr = malloc (sizeof (struct  mtx3x52_statusDataRanges));
    
	instrPtr -> triggerMode = 0;
	instrPtr -> val2 = 0;
	instrPtr -> val3 = 0;
	Fmt (instrPtr -> instrDriverRevision, "%s<%s",  MTX3x52_REVISION);
    
	if (( mtx3x52_status = viSetAttribute (instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr)) < 0)
		return  mtx3x52_status;                                       

	if (( mtx3x52_status = viWrite (instrSession, "*CLS;:FORM INT;:FORM:DINT OFF\r", 30, &retCnt)) < 0)
		return  mtx3x52_status;

	return  mtx3x52_status;    
}

/*===========================================================================*/
/* Function: Make data readable                                              */
/* Purpose:  This function fills output parameters with readable data        */
/*===========================================================================*/
ViStatus mtx3x52_makeDataReadable  (ViByte tabByte[], ViReal64 _VI_FAR waveformArray[], 
						ViPInt32 numberOfSamples, ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mtx3x52_status = VI_SUCCESS;
	ViInt32 i, index, start, type = 0;
	ViReal64 Offset, YIncrement = 0; 
	union 
		{
		ViByte  buffer[4];
		ViInt32 valSample; 
		} Buff;

#define OFFSET_ADC	393216		//0x60000 en hexa

	if ((start = FindPattern (tabByte, 0, 20, "#6200034", 0, 0)) != -1)
		type = 1;
		
	if (type)
		{
		*XIncrement = cmdTimeBase[tabByte[11]] / 50000.0;
		YIncrement = cmdVolt[tabByte[31]] / 32768.0;
		Offset = OFFSET_ADC;
		*numberOfSamples = 50000;
		start += 42;
		}
	else
		{
		/*-Find begining of data file--------------------------------------------*/    
		start = FindPattern (tabByte, 0, 200, "#6200", 0, 0);
		if (start == -1) 
			return VI_ERROR_MTX3x52_INVALID_FILE_CONTENT;
		start += 8;
	    index = FindPattern (tabByte, 0, 80, "SCALe", 0, 0);
		Scan(tabByte + index + 6, "%f", XIncrement);

		index = FindPattern (tabByte, 0, 90, "SIZE", 0, 0);
		Scan(tabByte + index + 5, "%i[b4]", numberOfSamples);
		 
		index = FindPattern (tabByte, 0, 200, "OFFSet", 0, 0);
		Scan(tabByte + index + 7, "%f", &Offset);
	
		index = FindPattern (tabByte, 100, 200, "SCALe", 0, 0);
		Scan(tabByte + index + 6, "%f", &YIncrement);
		}
	for (i=0; i<*numberOfSamples; i++)
		{
		Buff.buffer[0] = tabByte[start + i * 4 + 3];
		Buff.buffer[1] = tabByte[start + i * 4 + 2];
		Buff.buffer[2] = tabByte[start + i * 4 + 1];
		Buff.buffer[3] = tabByte[start + i * 4];
		waveformArray[i] = (Buff.valSample - Offset) * YIncrement;
		}
	return mtx3x52_status;
}

/*===========================================================================*/
/* Function: Check File Name                                                 */
/* Purpose:  This function checks if the file name is properly edited        */		
/*===========================================================================*/
ViBoolean mtx3x52_invalidFilename (ViString filename, ViInt16 drive, ViString type)
{
	ViInt16 startIndex = 0;
    
	if (!drive) 
		startIndex = FindPattern (filename, 0, -1, "\\", 0, 1) + 1;
	if (StringLength(filename + startIndex) > 12) 
		return VI_TRUE;
	startIndex = FindPattern (filename, 0, -1, ".", 0, 1);
	if (CompareStrings (filename, startIndex, type, 0, 0) != 0)  
		return VI_TRUE; 
		
	return VI_FALSE;
}

