#include "HandScope.h"
#include <ansi_c.h>
#include <utility.h>
#include <formatio.h>
#include <visa.h>		  


/*===================================================================================*/
#define mxOX50x2_REVISION   "Rev 1.0, 07/2011,CVI 9.0"/* Instrument driver revision  */
#define BUFFER_SIZE        512L                       /* File I/O buffer size        */
#define INVALID_SAMPLE     0xFFFF                     /* Sample validity criterion   */
#define MAX_MEM_SIZE       10300                      /* Memory needed for file I/O  */
#define INVALID_VALUE      1E100                      /* Invalid sample default value*/
#define TMO_VALUE          4000                       /* Time out value              */
#define TMO_TRANSFER       15000                      /* Time out value              */
#define HEADER_SIZE        200                        /* Size of binary data header  */
#define HEADER_FILE        20                         /* Size of binary file header  */
/*===================================================================================*/

/*=mxOX50x2==================================================================*/
/* LabWindows/CVI 9.0.1 Instrument Driver (VISA I/O)                         */
/* Original Release: Jully 2011                                              */
/* By:Metrix, Annecy, France                                                 */
/*                                                                           */                           
/* Metrix Technical Support: Olivier Lemaire                                 */
/*                                                                           */
/* Modification History: None                                                */
/*                                                                           */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/
/*-Vertical command arrays---------------------------------------------------*/
static ViString cmdVertCoupling[] = {"AC","DC","GRO"};       
static ViString cmdVoltDiv[] = {"20mV","40mV","80mV", "160mV","400mV","800mV",
						"1.6V","4V","8V", "16V","40V","80V", "160V","400V","800V","1.6kV"}; 
						
static ViReal64 cmdOffsetMax[] = {80E-3,80E-3,400E-3,400E-3,2.0,2.0,2.5,12.5,12.5,
						64.0,64.0,64.0,640.0,640.0,640.0,6400.0,6400.0,6400.0}; 
static ViString cmdInpSelect[]  = {"CHAN","MEM"};   

/*-Horizontal command arrays-------------------------------------------------*/
static ViString cmdTimeDiv[] = {"25E-9", "50E-9","125E-9", "250E-9", "500E-9", "1E-6", "2.5E-6", "5E-6", "10E-6", "20E-6",
								"50E-6", "100E-6", "200E-6", "500E-6", "1E-3", "2E-3", "5E-3",
								"10E-3", "20E-3", "50E-3", "100E-3", "200E-3", "500E-3","1",
								"2" ,"5", "10", "20", "50", "100", "200"};
                        
static ViString cmdDispTrace[] = {"A", "XY"}; 

/*-Trigger command arrays---------------------------------------------------*/
static ViString cmdTrigCoupling[] = {"AC","DC","ON"};       
static ViString cmdSlope[] = {"POS", "NEG"};
static ViString cmdPulse[] = {"", "INF", "SUP", "EQU"};
static ViString cmdSource[]= {"INT1", "INT2", "INT3", "INT4"}; 
static ViString cmdType[] = {"0","1","0"};  
static ViString cmdStandVid[] = {"525", "625"};
static ViString cmdSlopeVid[] = {"POS", "NEG"};   
static ViString CmdPulsTyp[] = {"EQU", "SUP", "INF"};
static ViString CmdTrigMode[] = {"EDGE", "PUL", "DEL", "EVEN", "TV", "REC", "CAPT"};
/*-Display command arrays----------------------------------------------------*/
static ViString cmddevice[]={"SCOP", "ANALYS", "MULT"};
static ViString cmdDisplayScreen[] = {"0", "15", "30", "60"};
static ViString cmdSystemStandby[] = {"30", "60", "240", "1440"};
static ViString cmdMaxminAcq[] = {"ENV", "NORM"};
static ViString cmdMode[] = {"NORM", "ENV"};       
static ViString cmdAverage[] = {"0", "2", "4", "16", "64"};       
/*-Read command arrays-------------------------------------------------------*/
static ViString cmdReadMeasure[] = {"MIN", "MAX", "PTP", "LOW", "HIGH", "AMPL", "AC","VOLT",
							"RISE:OVER", "FALL:OVER", "RTIME", "FTIME", "PWID","NWID", "PER", "FREQ", "PDUT",
							"PUL:COUN", "PHAS"};  
static ViReal64 cmdTimeBase[] = {1E-9,2E-9,5E-9,10E-9,25E-9, 50E-9, 125E-9, 250E-9, 500E-9,
							1E-6, 2.5E-6, 5E-6, 10E-6, 20E-6, 50E-6, 100E-6, 200E-6, 500E-6, 1E-3,
							2E-3, 5E-3, 10E-3, 20E-3, 50E-3, 100E-3, 200E-3, 500E-3, 1.0, 2.0, 5.0,
							10.0, 20.0, 50.0, 100.0, 200.0};
static ViReal64 cmdVolt[] = {2.5E-3,5E-3,10E-3,20E-3,50E-3,100E-3,200E-3,500E-3,1.0,
							2.0,5.0,10.0,20.0,50.0,100.0,200.0};
static ViReal64 cmdBandWith[] = {0, 1.5e6, 5e3};
static ViString cmdRef[] = {"", "INT1", "INT2"};
/*-Meter command arrays-------------------------------------------------------*/
static ViString MulCoup[] = {"AC", "DC", "ACDC"};
static ViString CmdMultMeas[] = {"VOLT", "POW", "POW3PN", "POW3P", "RES", "CONT", "CAPA", "DIOD", "RPM"};
static ViString CmdMultVolt[] = {"800E-3", "8.00", "80.0", "800.0"};
static ViString CmdMultOhm[]  = {"80.0", "800", "8.0E+3", "80.0E+3", "800E+3", "8.0E+6", "32.0E+6"};
static ViString CmdMulCapa[]  = {"5.00E-9", "50E-9", "500E-9", "5.00E-6", "50E-6", "500E-6", "5.00E-3"};
                                     
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/*  mxOX50x2_stringValPair is used in the  mxOX50x2_errorMessage function    */
/*===========================================================================*/
typedef struct   mxOX50x2_stringValPair
{
   ViStatus stringVal;
   ViPString stringName;
}   mxOX50x2_tStringValPair;
	 
struct mxOX50x2_statusDataRanges {
    ViInt16 triggerMode;
    ViInt16 val2;
    ViInt16 val3;
    ViChar instrDriverRevision[256];
    ViInt16 instrMode;
};
typedef struct mxOX50x2_statusDataRanges * mxOX50x2_instrRange;

ViInt16 TrigMode = 0 ;  
/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean mxOX50x2_invalidViBooleanRange (ViBoolean val);
ViBoolean mxOX50x2_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean mxOX50x2_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean mxOX50x2_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean mxOX50x2_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean mxOX50x2_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean mxOX50x2_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  mxOX50x2_makeDataReadable (ViByte tabByte[], ViReal64 waveformArray[], ViPInt32 numberOfSamples,
                                ViPReal64 XStart, ViPReal64 XIncrement, ViPReal64 YIncrement);
ViBoolean mxOX50x2_invalidFilename (ViString filename, ViString type);
ViStatus  mxOX50x2_instrStatus (ViSession instrSession);
ViStatus  mxOX50x2_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);
ViStatus  mxOX50x2_defaultInstrSetup (ViSession openInstrSession);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function:	Initialize                                                   */
/* Purpose:		This function opens the instrument, queries the instrument   */
/*				for its ID, and initializes the instrument to a known state  */
/*===========================================================================*/
ViStatus _VI_FUNC  mxOX50x2_init (ViRsrc resourceName, ViBoolean IDQuery, ViPSession instrSession)
{
	ViStatus  mxOX50x2_status = VI_SUCCESS;
	ViSession rmSession = 0;
	ViUInt32 retCnt = 0;
	ViByte wrBuffer[BUFFER_SIZE];
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte model[2];
	ViUInt16 interface;
    
	/*- Check input parameter ranges ----------------------------------------*/

	if ( mxOX50x2_invalidViBooleanRange (IDQuery))
		return VI_ERROR_PARAMETER2;

	/*- Open instrument session ---------------------------------------------*/

	if (( mxOX50x2_status = viOpenDefaultRM (&rmSession)) < 0)
		return  mxOX50x2_status;

	if (( mxOX50x2_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0) 
		{
		viClose (rmSession);
		return  mxOX50x2_status;
		}	

	/* Serial Interface initialized to 57600 bauds, 8 databits, 1 stopbit, no parity, no protocol*/
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 57600)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_DATA_BITS, 8)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_STOP_BITS, VI_ASRL_STOP_ONE)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_PARITY, VI_ASRL_PAR_NONE)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_NONE)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetBuf (*instrSession, VI_ASRL_IN_BUF|VI_ASRL_OUT_BUF, 5000)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0D)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_END_OUT, VI_ASRL_END_NONE)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viWrite (*instrSession, "\r", 1, &retCnt)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);

	/*- Configure VISA Formatted I/O ----------------------------------------*/
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE,VI_FLUSH_ON_ACCESS)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	if (( mxOX50x2_status = viSetAttribute (*instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);

	/*- Identification Query ------------------------------------------------*/
	if (IDQuery) 
		{
		Fmt (wrBuffer, "\r*IDN?\r");
		if (( mxOX50x2_status = viWrite (*instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
		if (( mxOX50x2_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
			return  mxOX50x2_status;
		if (FindPattern (rdBuffer, 0, -1, "OX50", 1, 0) == -1)
		return  mxOX50x2_initCleanUp (rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
		}

	/*- Send Default Instrument Setup ---------------------------------*/
	if (( mxOX50x2_status =  mxOX50x2_defaultInstrSetup (*instrSession)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	
	Fmt (wrBuffer, "FORM INT;FORM:DINT 1\r");
	if (( mxOX50x2_status = viWrite (*instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return  mxOX50x2_initCleanUp (rmSession, instrSession,  mxOX50x2_status);
	
	/*-Check instrument status & Return from function------------------------*/
	if (( mxOX50x2_status =  mxOX50x2_instrStatus(*instrSession)) < 0)
		return  mxOX50x2_status;

	return  mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Example                                                         */
/* Purpose:  This function is an example of this driver's functions use.     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_example (ViSession instrSession, ViInt16 channel,
					ViInt16 timebase, ViInt16 voltageRange, ViInt16 coupling, 
					ViReal64 data[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViReal64 YIncrement;
    
	/*- Check input parameter ranges ----------------------------------------*/    
	if (channel != 1 && channel != 4)
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViInt16Range (timebase, 0, 33))
		return VI_ERROR_PARAMETER3;
	if (mxOX50x2_invalidViInt16Range (voltageRange, 0, 17))
		return VI_ERROR_PARAMETER4;
	if (mxOX50x2_invalidViInt16Range (coupling, 0, 2))
		return VI_ERROR_PARAMETER5;
    
	/*- Configuring vertical settings----------------------------------------*/
	if((mxOX50x2_status = mxOX50x2_verticalSetup (instrSession, channel, VI_ON, coupling, 1, 
						voltageRange, 0.0, 0)) < 0)
		return mxOX50x2_status;
        
	/*- Setting timebase ----------------------------------------------------*/
	if((mxOX50x2_status = mxOX50x2_horizontalSetup (instrSession, timebase, 0.0)) < 0)
		return mxOX50x2_status;

	/*- Setting trigger mode to AUTO-----------------------------------------*/
	/*- mxOX50x2 starts acquiring data immediately after it is triggered-------*/
	if((mxOX50x2_status = mxOX50x2_setGeneralTrigger (instrSession, channel, VI_ON, 0, VI_OFF, 1, 0.0, 200e-9)) < 0)
		return mxOX50x2_status;
        
	/*- Sending single-shot trigger in wait mode-----------------------------*/
	/*- This function will return as long as the acquisition is running------*/
	if((mxOX50x2_status = mxOX50x2_runAcquisition (instrSession, VI_ON)) < 0)
		return mxOX50x2_status;
		
    Delay (5.0);
        
	/*- Reads trace from the instrument--------------------------------------*/ 
	if((mxOX50x2_status = mxOX50x2_readChToArr (instrSession, channel - 1, data, numberOfSamples, XStart, XIncrement, &YIncrement)) < 0)
		return mxOX50x2_status; 
                   
	return mxOX50x2_status;
}
 
/*===========================================================================*/
/* Function: Configure mxOX50x2 Vertical Settings                            */
/* Purpose:  This function configures the vertical settings for a specified  */
/*           channel                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_verticalSetup (ViSession instrSession, ViInt16 input,
					ViBoolean state, ViInt16 coupling, ViInt16 probe, 
					ViInt16 sense, ViReal64 offset, ViInt16 band)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(input, 1, 2))
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViBooleanRange(state))
		return VI_ERROR_PARAMETER3;
	if (mxOX50x2_invalidViInt16Range(coupling, 0, 2))
		return VI_ERROR_PARAMETER4;
	if (probe != 1 && probe != 10 && probe != 20 && probe != 50 && probe != 100
		&& probe != 200 && probe != 1000)
		return VI_ERROR_PARAMETER5;
	if (mxOX50x2_invalidViInt16Range(sense, 1, 15))
		return VI_ERROR_PARAMETER6;
	if (mxOX50x2_invalidViReal64Range(offset,-cmdOffsetMax[sense],cmdOffsetMax[sense]))
		return VI_ERROR_PARAMETER7;
	if (mxOX50x2_invalidViInt16Range(band, 0, 2))
		return VI_ERROR_PARAMETER8;
    
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] 1\r", input);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status; 
	Fmt (wrBuffer, "DISP:TRAC:STAT%i[b2] %i[b2]\r", input, state);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status; 
	Fmt (wrBuffer, "INP%i[b2]:COUP %s\r", input, cmdVertCoupling[coupling]);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status; 
	Fmt (wrBuffer, "SENS:BAND%i[b2] %f\r", input, cmdBandWith[band]);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:PTP %s\r", input, cmdVoltDiv[sense]);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status; 
	Fmt (wrBuffer, "VOLT%i[b2]:RANG:OFFS %f\r", input, offset);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status; 
	Fmt (wrBuffer, "DISP:TRAC:Y:PDIV%i[b2] %i[b2]\r", input, probe);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status; 
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Configure mxOX50x2 Horizontal Settings                          */
/* Purpose:  This function configures the horizontal settings of the         */
/*           instrument                                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_horizontalSetup (ViSession instrSession,
					ViInt16 range, ViReal64 pretrg) 
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViReal64 convBuf;
	ViByte pretrgBuf[BUFFER_SIZE];
	ViInt32 retCntptr;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(range, 0, 30))
		return VI_ERROR_PARAMETER2;
      
	/*----------------------- time bases ------------------------------------*/
	Fmt (pretrgBuf, "DISP:TRAC:X:PDIV %s\r", cmdTimeDiv[range]);
	if ((mxOX50x2_status = viWrite (instrSession, pretrgBuf, strlen(pretrgBuf), &retCntptr)) < 0)
		return mxOX50x2_status; 
        
	/*-Check input parameter ranges and test pretrig-------------------------*/    
	Fmt(&convBuf, "%f<%s", cmdTimeDiv[range]);
	if (mxOX50x2_invalidViReal64Range(pretrg, -9.5 * convBuf, 10 * convBuf))
		return VI_ERROR_PARAMETER3; 
  
	/*----------------  pretrig --------------------------------------------*/       
	Fmt (pretrgBuf, "SWE:OFFS:TIME %f\r", pretrg);
	if ((mxOX50x2_status = viWrite (instrSession, pretrgBuf, strlen(pretrgBuf), &retCntptr)) < 0)
		return mxOX50x2_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Configure mxOX50x2 Horizontal Display                           */
/* Purpose:  This function configures the horizontal display                 */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_horizontalDisplay (ViSession instrSession, ViInt16 trace)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(trace,0,1))
		return VI_ERROR_PARAMETER2;
   
	Fmt (wrBuffer,"DISP:TRAC:FORM %s\r",cmdDispTrace[trace]);
	if ((mxOX50x2_status = viWrite (instrSession,wrBuffer,strlen(wrBuffer),&retCnt)) < 0)
		return mxOX50x2_status;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
     
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: General Trigger                                                 */
/* Purpose:  This function sets general parameters for trigger               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_setGeneralTrigger (ViSession instrSession, ViInt16 channel,
								ViBoolean slope, ViInt16 pulse, ViReal64 pulseWidth,
								ViInt16 type, ViReal64 level, ViInt16 filter)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViReal64 convBuf;
	ViInt32 retCnt;
	ViInt32 Sequence = 1;
	
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(channel,1,2))
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViBooleanRange(slope))
		return VI_ERROR_PARAMETER3;
	if (mxOX50x2_invalidViInt16Range(pulse,0,3))
		return VI_ERROR_PARAMETER4;
	if (mxOX50x2_invalidViReal64Range(pulseWidth,20E-9,20.0))
		return VI_ERROR_PARAMETER5;
	if (mxOX50x2_invalidViInt16Range(type,0,2))
		return VI_ERROR_PARAMETER6;
	if (mxOX50x2_invalidViReal64Range(level,-800,800))
		return VI_ERROR_PARAMETER7;
	if (mxOX50x2_invalidViInt16Range(filter,0,3))
		return VI_ERROR_PARAMETER8;
        
	if (pulse)
		Sequence = 2;
	
	/*-Set slope,coupling,level,holdoff and filter parameters----------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,"TRIG:SEQ%i:SOUR INT%hd\r", Sequence, channel)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viPrintf (instrSession,"TRIG:SEQ%i:LEV %lf\r", Sequence, level)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viPrintf (instrSession,"TRIG:SEQ%i:SLOP %s\r", Sequence, cmdSlope[slope])) < 0)
		return mxOX50x2_status;
        
	if (pulse)
		{
		Fmt (wrBuffer, "TRIG:SEQ2:TYP %s\r", cmdPulse[pulse]);
		if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
			return mxOX50x2_status;

		Fmt (wrBuffer, "TRIG:SEQ2:DEL %f\r", pulseWidth);
		if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
			return mxOX50x2_status;
		}

	switch (filter)
		{
		case 0 :
			Fmt (wrBuffer, "TRIG:SEQ%i:HYST 1\rTRIG:SEQ%i:FILT:HPAS 0\rTRIG:SEQ%i:FILT:HPAS 0\r", Sequence, Sequence, Sequence);
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			break;
		case 1 :
			Fmt (wrBuffer, "TRIG:SEQ%i:FILT:LPAS 1\r", Sequence);
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			break;	
		case 2 :
			Fmt (wrBuffer, "TRIG:SEQ%i:FILT:HPAS 1\r", Sequence);
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			break;	
		case 3 :
			Fmt (wrBuffer, "TRIG:SEQ%i:HYST %i\r", Sequence, filter);
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			break;	
		}

	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;

	/*-Set trigger type and mode parameters----------------*/
/*	switch (type)
		{
		case 0 :
			TrigMode = 0;
			Fmt (wrBuffer, "TRIG:SEQ1:ATRIG %s\r", cmdType[type]);
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			Fmt (wrBuffer, "INIT:CONT:NAME EDGE,1\r");
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			break;
		
		case 1 :
			TrigMode = 1;
			Fmt (wrBuffer, "TRIG:SEQ1:ATRIG %s\r", cmdType[type]);
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			break;

		case 2 :
			Fmt (wrBuffer, "INIT:NAME EDGE\r");
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status; 
			TrigMode = 2;
			break;
		}
*/
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: PULSE Trigger                                                   */
/* Purpose:  This function sets Pulsate trigger type                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_setPulsTrigger (ViSession instrSession, ViInt16 pulsType,
                    		ViReal64 pulse)
{                                
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViReal64 convBuf;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViBoolean runStat;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(pulsType,0,2))
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViReal64Range(pulse,20E-9,20.0))
		return VI_ERROR_INSTR_PARAMETER9;

   	if(TrigMode == 0)
		{
		Fmt (rdBuffer ,"TRIG:RUN:STAT?\r");
		if ((mxOX50x2_status = viWrite (instrSession,rdBuffer,strlen(rdBuffer), &retCnt)) < 0)
			return mxOX50x2_status;
		if ((mxOX50x2_status = viScanf (instrSession,"%hd", &runStat)) < 0)
			return mxOX50x2_status;
		Fmt (rdBuffer, "INIT:CONT:NAME PUL,%i[b1]\r", runStat);
		if ((mxOX50x2_status = viWrite (instrSession, rdBuffer, strlen(rdBuffer), &retCnt)) < 0)
			return mxOX50x2_status;
		}
	 else if(TrigMode == 2)
	 	{
		Fmt (rdBuffer, "INIT:NAME PUL\r");
		if ((mxOX50x2_status = viWrite (instrSession, rdBuffer, strlen(rdBuffer), &retCnt)) < 0)
			return mxOX50x2_status; 
	 	}
    
	/*-Set source and levels for edge trigger----------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession, "TRIG:TYP %s\r", CmdPulsTyp[pulsType])) < 0)
		return mxOX50x2_status;
	Fmt (rdBuffer,"TRIG:SEQ2:DEL %f\r",pulse);
	if ((mxOX50x2_status = viWrite (instrSession, rdBuffer, strlen(rdBuffer), &retCnt)) < 0)
		return mxOX50x2_status; 

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}


/*===========================================================================*/
/* Function: Display Options                                                 */
/* Purpose:  This function sets display options.                             */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_setOptionDisplay (ViSession instrSession, ViInt16 device_mode)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(device_mode,0,2))
		return VI_ERROR_PARAMETER2;

	/*-Set option display parameters-----------------------------------------*/
	Fmt (wrBuffer, "DEV:MOD %s\r", cmddevice[device_mode]);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
	/*-Check instrument status & Return from function------------------------*/				   
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
      
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Display Waveform                                                */
/* Purpose:  This function sets waveform display.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_setWaveformDisplay (ViSession instrSession,ViInt16 peakDetect,
					ViInt16 average_rate, ViInt16 envelope)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(peakDetect, 0,1))
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViInt16Range(average_rate, 0, 4))
		return VI_ERROR_PARAMETER3;
	if (mxOX50x2_invalidViInt16Range(envelope, 0, 1))
		return VI_ERROR_PARAMETER4;

	/*-Set waveform display parameters---------------------------------------*/
	Fmt (wrBuffer, "AVER:TYPE %s\r", cmdMaxminAcq[peakDetect]);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
	Fmt (wrBuffer, "DISP:TRAC:MODE %s\r", cmdMode[envelope]);
		if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
	Fmt (wrBuffer, "AVER:COUN %s\r", cmdAverage[average_rate]);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
	
    /*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Set Cursors                                                     */
/* Purpose:  This function sets mxOX50x2 cursors.                            */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_setCursors (ViSession instrSession, ViInt16 cursor,
					ViInt16 reference, ViReal64 t1, ViReal64 t2)
{                              
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(cursor,0,1))
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViInt16Range(reference,1,2))
		return VI_ERROR_PARAMETER3;
	if (mxOX50x2_invalidViReal64Range(t1,0,2000.0))
		return VI_ERROR_PARAMETER4;
	if (mxOX50x2_invalidViReal64Range(t2,0,2000.0))
		return VI_ERROR_PARAMETER5;
	 
	/*-Set cursors-----------------------------------------------------------*/    
	Fmt (wrBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
    
	switch(cursor)
		{
		case 0 :
			Fmt (wrBuffer, "DISP:CURS:STAT OFF\r");
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			break;
			
		case 1 :
			Fmt (wrBuffer, "DISP:CURS:STAT ON\r");
			if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
				return mxOX50x2_status;
			if ((mxOX50x2_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS %lf\r",t1)) < 0)
				return mxOX50x2_status;
			if ((mxOX50x2_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS %lf\r",t2)) < 0)
				return mxOX50x2_status;
			break;
		
		}

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Run Autoset                                                     */
/* Purpose:  This function performs an automatic setup of the mxOX50x2.      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_runAutoset (ViSession instrSession)
{                              
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViInt16 i,state;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	Fmt (wrBuffer, "AUTOS:EXE\r");
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Run Acquisition                                                 */
/* Purpose:  This function performs an acquisition session.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_runAcquisition (ViSession instrSession, ViBoolean run)
{                              
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViInt32 retCnt;
	ViByte rdBuffer[BUFFER_SIZE];
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViBooleanRange(run))
		return VI_ERROR_PARAMETER3;
    
	/*-Get current trigger mode----------------------------------------------*/
	Fmt (rdBuffer, "TRIG:RUN:STAT %i[b2]\r", run);
	if ((mxOX50x2_status = viWrite (instrSession, rdBuffer, strlen(rdBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
    return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Read Cursors                                                    */
/* Purpose:  This function reads mxOX50x2 cursors.                           */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_readCursors (ViSession instrSession, ViInt16 reference, ViPReal64 t1, 
							ViPReal64 t2, ViPReal64 dt, ViPReal64 v1, ViPReal64 v2, ViPReal64 dv)
{                              
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViInt16 valtphase;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(reference,1,2))
		return VI_ERROR_PARAMETER2;

	/*-Display cursor menu---------------------------------------------------*/
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mxOX50x2_status = viWrite (instrSession, rdBuffer, strlen(rdBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
        
	/*-Set cursor positions--------------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,"DISP:CURS:TIME1:POS?\r")) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viScanf (instrSession,"%lf",t1)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viPrintf (instrSession,"DISP:CURS:TIME2:POS?\r")) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viScanf (instrSession,"%lf",t2)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viPrintf (instrSession,"DISP:CURS:VOLT1:POS?\r")) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viScanf (instrSession,"%lf",v1)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viPrintf (instrSession,"DISP:CURS:VOLT2:POS?\r")) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viScanf (instrSession,"%lf",v2)) < 0)
		return mxOX50x2_status;
		
	/*-Calculate 'delta t' & 'delta v'---------------------------------------*/
	*dt = *t2 - *t1;
	*dv = *v2 - *v1;
        
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Read Measurement                                                */
/* Purpose:  This function reads a specified measurement.                    */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_readMeasurement (ViSession instrSession, ViInt16 measure,
					ViInt16 reference, ViPReal64 reading)
{                              
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(measure,0,18))
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViInt16Range(reference,1,2))
		return VI_ERROR_PARAMETER3;

	/*-Set hardcopy parameters-----------------------------------------------*/
	Fmt (rdBuffer, "DISP:CURS:REF %s\r", cmdRef[reference]);
	if ((mxOX50x2_status = viWrite (instrSession, rdBuffer, strlen(rdBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viPrintf (instrSession,"MEAS:%s? %s\r",cmdReadMeasure[measure],cmdRef[reference])) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = Scan (rdBuffer,"%s>%f",reading))!=1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
     
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Read Channel To Array                                           */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_readChToArr (ViSession instrSession, ViInt16 source,			   
					ViReal64 waveformArray[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement, ViPReal64 YIncrement)
{   
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,size, index;
	ViInt16 tstRes;
	ViUInt16 interface;
	static ViByte tabByte[MAX_MEM_SIZE];
	ViByte RunStat;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(source,1,2))
		return VI_ERROR_PARAMETER2;
    
	/*-Check if source is active---------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", source + 1)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mxOX50x2_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_mxOX50x2_CHAN_NOT_ACTIVE;
    
	/*-Set to Stop to perform transfer ----------------------------------*/
    viWrite (instrSession, "TRIG:RUN:STAT?\r", 15, &retCnt);
    viScanf (instrSession, "%hd", &RunStat);
    if (RunStat)
	    viWrite (instrSession, "TRIG:RUN:STAT 0\r", 16, &retCnt);

	/*-Modify Term Char--------------------------------------------------*/
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
		return mxOX50x2_status;

	/*-Set data format and start transfert-----------------------------------*/    
	if ((mxOX50x2_status = viPrintf (instrSession,"TRAC? INT%hd\r", source)) < 0)
		return mxOX50x2_status;

	/* Read 11 characters (#an header) to define the length of the binary block following */
	if ((mxOX50x2_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
		return mxOX50x2_status;
	index = FindPattern (tabByte, 0, HEADER_SIZE, "#", 0, 0);
	headerSize = tabByte[index + 1] - 0x30;
	Scan(tabByte + index + 2, "%i[b4]", &size);

	/*-Modify Protocol for binary file transfers-------------------------*/
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_TRUE)) < 0)
		return mxOX50x2_status;

	if ((mxOX50x2_status = viRead (instrSession, tabByte + HEADER_SIZE, index + size + 7 + headerSize - HEADER_SIZE, &retCnt)) < 0)
		return mxOX50x2_status;

	/*-Reset Protocol for binary file transfers-------------------------*/
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
		return mxOX50x2_status;

	/*-Set to Run if Stop forced ----------------------------------------*/
    if (RunStat)
	    viWrite (instrSession, "TRIG:RUN:STAT 1\r", 16, &retCnt);

	/*-Make data readable----------------------------------------------------*/
	if ((mxOX50x2_status = mxOX50x2_makeDataReadable(tabByte,waveformArray,numberOfSamples,XStart,XIncrement,YIncrement)) < 0)
		return mxOX50x2_status;
                            
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Read Channel To File                                            */
/* Purpose:  This function reads a trace from the instrument.                */
/*           Data is stored in a file                                        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_readChToFile ( ViSession instrSession, ViInt16 channel,
					ViString filename, ViBoolean overwrite)
{                        
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 fileSize, retCnt, size, index;													
	ViByte tabByte[MAX_MEM_SIZE], RdBuffer[50];
	ViInt16 tstRes;
	ViString filenameTemp = "TRC";

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(channel,1,2))
		return VI_ERROR_PARAMETER2;								 
	if (mxOX50x2_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER4;
        
	/*-Check if source is active---------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,":DISP:TRAC:STAT%hd?\r", channel)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, tabByte, 5, &retCnt)) < 0)
		return mxOX50x2_status;
	Scan (tabByte,"%i[b2]", &tstRes);
	if (tstRes != 1) 
		return VI_ERROR_mxOX50x2_CHAN_NOT_ACTIVE;
    
	/*-Check filename--------------------------------------------------------*/
	if (mxOX50x2_invalidFilename (filename,".TRC"))
		return VI_ERROR_PARAMETER4;
    
	/*-Store current trace into a PC file------------------------------------*/
	/*-Check overwrite---------------------------------------------------*/
	if (GetFileInfo(filename,&fileSize)  && !overwrite) 
		return VI_ERROR_mxOX50x2_EXISTING_FILE;

	if (GetFileInfo (filename,&fileSize)  && overwrite) 
		DeleteFile (filename);

	/*-Perform transfer--------------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,"MMEM:STOR:TRAC INT%hd,\"%s\"\r",
				channel, filenameTemp )) < 0)
		return mxOX50x2_status;

	/*-Recover the file name---------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,"MMEM:STOR:TRAC?\r")) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, RdBuffer, 20, &retCnt)) < 0)
		return mxOX50x2_status;
	if (retCnt > 2)
		RdBuffer[retCnt - 2] = 0;

	/*-Temp file transfer------------------------------------------------*/
	if ((mxOX50x2_status = mxOX50x2_fileTransferToComputer (instrSession, RdBuffer, VI_ON)) < 0)
		return mxOX50x2_status;

	/*-Rename Temp file--------------------------------------------------*/
	RenameFile (RdBuffer, filename);
		
	/*-Delete Temp file--------------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,"MMEM:DEL \"%s\"\r", RdBuffer)) < 0)
		return mxOX50x2_status;

	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Read File To Array                                              */
/* Purpose:  This function reads a file from the PC or from the instrument   */
/*           Data is stored in an array                                      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_readFileToArr (ViSession instrSession, ViInt16 drive,
					ViString filename, ViReal64 waveformArray[], ViPInt32 numberOfSamples, 
					ViPReal64 XStart, ViPReal64 XIncrement, ViPReal64 YIncrement)
{   
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViInt32 retCnt,fileSize, size, index;
	static ViByte tabByte[MAX_MEM_SIZE];
	ViUInt16 interface;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(drive,0,1))
		return VI_ERROR_PARAMETER2;
	        
	/*-Check filename--------------------------------------------------------*/
	if (mxOX50x2_invalidFilename (filename,".TRC"))
		return VI_ERROR_PARAMETER3;											   
        
	/*-Read file from mxOX50x2-----------------------------------------------*/
	if (drive) 
		{
		/*-Check filename----------------------------------------------------*/
		if ((mxOX50x2_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
			return mxOX50x2_status;
		if ((mxOX50x2_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
			return mxOX50x2_status;
		if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
			return VI_ERROR_mxOX50x2_FILE_NOT_FOUND;
        
		/*-Modify Protocol for binary file transfers-------------------------*/
		if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
			return mxOX50x2_status;

		/*-Perform transfer--------------------------------------------------*/
		if ((mxOX50x2_status = viPrintf (instrSession,"MMEM:DATA? \"%s\"\r",filename)) < 0)
			return mxOX50x2_status;
		
		/* Read 11 characters (#an header) to define the length of the binary block following */
		if ((mxOX50x2_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
			return mxOX50x2_status;
		index = FindPattern (tabByte, 0, -1, "#", 0, 0);
		headerSize = tabByte[index + 1] - 0x30;
		Scan(tabByte + index + 2, "%i[b4]", &size);
		if ((mxOX50x2_status = viRead (instrSession, tabByte + HEADER_SIZE, size + 4 + headerSize - HEADER_SIZE, &retCnt)) < 0)
			return mxOX50x2_status;
			
		retCnt += HEADER_SIZE;

		/*-Reset Protocol for binary file transfers----------------------*/
		if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
			return mxOX50x2_status;
    	}
	/*-Read file from PC-----------------------------------------------------*/
	else 
		{
		/*-Check filename----------------------------------------------------*/
		if (!GetFileInfo(filename,&fileSize)) 
			return VI_ERROR_mxOX50x2_FILE_NOT_FOUND;
        if ((FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, MAX_MEM_SIZE,MAX_MEM_SIZE,
			VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_BINARY)) < 0)
		return VI_ERROR_INSTR_FILE_OPEN;
		}
		
	/*-Make data readable----------------------------------------------------*/
	if ((mxOX50x2_status = mxOX50x2_makeDataReadable(tabByte,waveformArray,numberOfSamples, XStart, XIncrement, YIncrement)) < 0)
		return mxOX50x2_status;
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
        
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Write File To Instrument Display Memory                         */
/* Purpose:  This function fills the instrument display memory.              */
/*           Data are in the specified file.                                 */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_writeFileToMem(ViSession instrSession, ViString filename)
{   
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViUInt32 retCnt;
	static ViByte tabByte[MAX_MEM_SIZE];

	/*-Check filename--------------------------------------------------------*/
	if (mxOX50x2_invalidFilename (filename,".TRC"))
		return VI_ERROR_PARAMETER3;
    
	/*-Check filename--------------------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,"MMEM:CAT?\r")) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
		return mxOX50x2_status;
	if (FindPattern (tabByte, 0, -1, filename , 0, 0) < 0)
		return VI_ERROR_mxOX50x2_FILE_NOT_FOUND;

	/*-Perform transfer------------------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,"MMEM:LOAD:TRAC \"%s\"\r", filename)) < 0)
		return mxOX50x2_status;
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
    
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Save/Recall Setup From/To Instrument                            */
/* Purpose:  This function allows you to save the current setup of the       */
/*           instrument or to recall a setup stored in a file.               */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_saveSetup (ViSession instrSession, ViBoolean mode, ViString filename, 
										ViBoolean overwrite)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViInt16 headerSize;
	ViUInt32 retCnt, fileSize;
	static ViByte string[2000],tabByte[MAX_MEM_SIZE];
	ViUInt16 interface;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViBooleanRange(mode))
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER4;

	/*-Check filename--------------------------------------------------------*/
	if (mxOX50x2_invalidFilename (filename,".CFG"))
		return VI_ERROR_PARAMETER3;

	/*-Save setup from mxOX50x2----------------------------------------------*/
	if (mode) 	
		{
		/*-Target drive is PC------------------------------------------------*/
		/*-Check overwrite-----------------------------------------------*/    
		if (GetFileInfo(filename,&fileSize)  && !overwrite) 
			return VI_ERROR_mxOX50x2_EXISTING_FILE;
        
		/*-Perform transfer----------------------------------------------*/    
		if ((mxOX50x2_status = viPrintf (instrSession,"SYST:SET?\r")) < 0)
			return mxOX50x2_status;
        
		/* Read 11 characters (#an header) to define the length of the binary block following */
		if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
		    return mxOX50x2_status;
		if ((mxOX50x2_status = viRead (instrSession, tabByte, HEADER_SIZE, &retCnt)) < 0)
		    return mxOX50x2_status;

		headerSize = tabByte[1] - 0x30;
		Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);

		/*-Modify Protocol for binary file transfers-------------------------*/
		if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_TRUE)) < 0)
			return mxOX50x2_status;

		if ((mxOX50x2_status = viRead (instrSession, tabByte + HEADER_SIZE, fileSize+4+headerSize-HEADER_SIZE, &retCnt)) < 0)
		     return mxOX50x2_status;

		/*-Reset Protocol for binary file transfers-------------------------*/
		if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
		    return mxOX50x2_status;
		if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
			return mxOX50x2_status;

		retCnt += HEADER_SIZE;

		/*-Fill PC file--------------------------------------------------*/    
		if ((ArrayToFile (filename, tabByte, VAL_UNSIGNED_CHAR, retCnt - 2,retCnt - 2, VAL_GROUPS_TOGETHER,
				VAL_GROUPS_AS_ROWS,VAL_CONST_WIDTH, 4, VAL_BINARY,VAL_TRUNCATE)) < 0) 
			return VI_ERROR_INSTR_FILE_WRITE;
		}
    
	/*-Recall setup to mxOX50x2----------------------------------------------*/
	else 
		{
		/*-Source drive is PC------------------------------------------------*/
		/*-Check file name-----------------------------------------------*/
		if (!GetFileInfo(filename,&fileSize)) 
			return VI_ERROR_mxOX50x2_FILE_NOT_FOUND;
		/*-Read PC file--------------------------------------------------*/
		if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, 20, 20,VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
			return VI_ERROR_INSTR_FILE_OPEN;
		Scan(tabByte,"%s[i2]>%i[b4]",&fileSize);
		headerSize = 6;
		if (FileToArray (filename, tabByte, VAL_UNSIGNED_CHAR, headerSize + fileSize, 
				headerSize + fileSize, VAL_GROUPS_TOGETHER,VAL_GROUPS_AS_ROWS, VAL_BINARY) < 0)
			return VI_ERROR_INSTR_FILE_OPEN;
		/*-Perform transfer----------------------------------------------*/
		Fmt(string,"%s<SYST:SET %s[t-w*q]\r", headerSize + fileSize, tabByte);
		if ((mxOX50x2_status = viWrite (instrSession, string, NumFmtdBytes (), &retCnt)) < 0) 
			return mxOX50x2_status;
		
		Delay(1.5);	
		}																  
    
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;
	 
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: File Transfer                                                   */ 
/* Purpose:  This function allows you to transfer a file from HandScope to   */
/*           computer.                                                       */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_fileTransferToComputer (ViSession instrSession, ViString filename, ViBoolean overwrite)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte trueFilename[MAX_FILENAME_LEN];
	ViInt16 headerSize;
	ViInt32 retCnt,fileSize, size, index;
	static ViByte tabByte[8*MAX_MEM_SIZE],tempString[20];
	ViUInt16 interface;

	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViBooleanRange(overwrite))
		return VI_ERROR_PARAMETER3;
  
	/*-Check filename--------------------------------------------------------*/
	if ((FindPattern (filename, 0, -1, ".", 0, 0) > 8) && (strlen(filename) > 20))
		return VI_ERROR_PARAMETER2;

	/*-Check filename----------------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession, "MMEM:CAT?\r")) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, tabByte, MAX_MEM_SIZE, &retCnt)) < 0)
		return mxOX50x2_status;
	if (FindPattern (tabByte, 0, -1, filename, 0, 0) < 0)
		return VI_ERROR_mxOX50x2_FILE_NOT_FOUND;

	/*-Check overwrite---------------------------------------------------*/    
	if (GetFileInfo(filename, &fileSize)  && !overwrite) 
		return VI_ERROR_mxOX50x2_EXISTING_FILE;

	/*-Modify Time Out for big file transfers----------------------------*/
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_TRANSFER)) < 0)
		return mxOX50x2_status;

	/*-Perform transfer--------------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession, "MMEM:DATA? \"%s\"\r", filename)) < 0)
		return mxOX50x2_status;

	/*-Modify Protocol for binary file transfers-------------------------*/
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_NONE)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_TRUE)) < 0)
		return mxOX50x2_status;

	/* Read 11 characters (#an header) to define the length of the binary block following */
	if ((mxOX50x2_status = viRead (instrSession, tabByte, HEADER_FILE, &retCnt)) < 0)
		return mxOX50x2_status;

	headerSize = tabByte[1] - 0x30;
	Scan(tabByte, "%s[i2]>%i[b4]", &fileSize);

	if ((mxOX50x2_status = viRead (instrSession, tabByte+HEADER_FILE, fileSize+4+headerSize-HEADER_FILE, &retCnt)) < 0)
		return mxOX50x2_status;

	/*-Reset Protocol for binary file transfers-------------------------*/
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_ASRL_END_IN, VI_ASRL_END_TERMCHAR)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_SUPPRESS_END_EN, VI_FALSE)) < 0)
		return mxOX50x2_status;

	retCnt += HEADER_FILE;

	/*-Reset Time Out ---------------------------------------------------*/
	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return mxOX50x2_status;

	/*-Parse non ".TRC" or ".CFG" files----------------------------------*/
	if ((FindPattern(filename,0,-1,"TRC",0,0)==-1) && (FindPattern(filename, 0, -1, "CFG", 0, 0)==-1))
		headerSize = tabByte[1] - 0x30 + 2; 
	else
		headerSize = 0;

	/*-Fill PC file------------------------------------------------------*/    
	if ((ArrayToFile (filename, tabByte + headerSize, VAL_UNSIGNED_CHAR, retCnt - headerSize - 2, 1, 
	VAL_GROUPS_TOGETHER, VAL_GROUPS_AS_ROWS, VAL_CONST_WIDTH, 4, VAL_BINARY, VAL_TRUNCATE)) < 0) 
		return VI_ERROR_INSTR_FILE_WRITE;
	
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;

	if ((mxOX50x2_status = viPrintf (instrSession, "%s", writeBuffer)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_readInstrData (ViSession instrSession, ViInt32 numBytes,
							ViChar rdBuf[], ViPInt32 bytesRead)

{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	*bytesRead = 0L;

	if ((mxOX50x2_status = viRead (instrSession, rdBuf, BUFFER_SIZE, bytesRead)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mxOX50x2_reset (ViSession instrSession)
{
	ViUInt32 retCnt = 0;
	ViStatus  mxOX50x2_status = VI_SUCCESS;

	/*-Initialize the instrument to a known state----------------------------*/
	if (( mxOX50x2_status = viWrite (instrSession, "\r*RST\r", 6, &retCnt)) < 0)
		return  mxOX50x2_status;

	if (( mxOX50x2_status =  mxOX50x2_defaultInstrSetup (instrSession)) < 0)  
		return  mxOX50x2_status;
        
	return  mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result.                                                     */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_selfTest (ViSession instrSession, ViPInt16 testResult,
								ViChar testMessage[])
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE,20 * TMO_VALUE)) < 0)
		return mxOX50x2_status; 
    
	if ((mxOX50x2_status = viWrite (instrSession, "*TST?\r", 6, &retCnt)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return mxOX50x2_status;
	if (Scan (rdBuffer, "%i[b1]", testResult) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	if (*testResult) 
		Fmt(testMessage, "%s<Self test failed");
	else 
		Fmt(testMessage, "%s<No Error");  

	if ((mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_TMO_VALUE, 
		TMO_VALUE)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_errorQuery (ViSession instrSession, ViPInt32 errCode,
							ViChar errMessage[])
{
	ViUInt32 retCnt = 0;
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mxOX50x2_status = viWrite (instrSession, "SYST:ERR?\r", 10, &retCnt)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mxOX50x2_status;

	if (Scan (rdBuffer, "%i[b4],%s[t10]", errCode,errMessage) != 2)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_errorMessage (ViSession instrSession, ViStatus errorCode,
							ViChar errMessage[])
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViInt16 i;
    
	static mxOX50x2_tStringValPair statusDescArray[] = {
		{VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
		{VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
		{VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
		{VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
		{VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
		{VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
		{VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
		{VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
		{VI_ERROR_INSTR_PARAMETER9,   "ERROR: Parameter 9 out of range"},
		{VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
		{VI_ERROR_INSTR_FILE_OPEN,    "ERROR: Opening the specified file"},
		{VI_ERROR_INSTR_FILE_WRITE,   "ERROR: Writing to the specified file"},
		{VI_ERROR_INSTR_INTERPRETING_RESPONSE,           "ERROR: Interpreting the instrument's response"},
		{VI_ERROR_mxOX50x2_ERROR,                  "ERROR: Instrument specific error"},
		{VI_ERROR_mxOX50x2_INCONSISTENT_PARAMETER3,"ERROR: Parameter 3 is INCONSISTENT"},
		{VI_ERROR_mxOX50x2_INCONSISTENT_PARAMETER5,"ERROR: Parameter 5 is INCONSISTENT"},
		{VI_ERROR_mxOX50x2_INCONSISTENT_PARAMETER6,"ERROR: Parameter 6 is INCONSISTENT"},
		{VI_ERROR_mxOX50x2_INCONSISTENT_PARAMETER7,"ERROR: Parameter 7 is INCONSISTENT"},
		{VI_ERROR_mxOX50x2_AUTOTEST_PROBLEM,       "ERROR: Performing a self-test"}, 
		{VI_ERROR_mxOX50x2_CHAN_NOT_ACTIVE,        "ERROR: Channel is not active"},
		{VI_ERROR_mxOX50x2_NO_CHAN_ENABLED,        "ERROR: No channel is enabled"},
		{VI_ERROR_mxOX50x2_FILE_NOT_FOUND,         "ERROR: File not available on support"},
		{VI_ERROR_mxOX50x2_EXISTING_FILE,          "ERROR: Filename already used"},
		{VI_ERROR_mxOX50x2_INVALID_FILE_CONTENT,   "ERROR: File content is invalid"},
		{VI_NULL, VI_NULL}  };
	          
	mxOX50x2_status = viStatusDesc (instrSession, errorCode, errMessage);
	if (mxOX50x2_status == VI_WARN_UNKNOWN_STATUS) 
		{
		for (i=0; statusDescArray[i].stringName; i++) 
			{
			if (statusDescArray[i].stringVal == errorCode) 
				{
				Fmt (errMessage, "%s<%s", statusDescArray[i].stringName);
				return (VI_SUCCESS);
				}
			}
		Fmt (errMessage, "%s<Unknown Error 0x%x[uw8p0]", errorCode);
		return (VI_WARN_UNKNOWN_STATUS);
		}
	mxOX50x2_status = VI_SUCCESS;
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_revisionQuery (ViSession instrSession,
					ViChar driverRev[], ViChar instrRev[])
{
	ViUInt32 retCnt = 0;
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];

	if ((mxOX50x2_status = viWrite (instrSession, "*IDN?\r", 6, &retCnt)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return mxOX50x2_status;
	if (Scan (rdBuffer, "%s[t10]", instrRev) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
	Fmt (driverRev, "%s<%s", mxOX50x2_REVISION);

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: MeterConfig                                                     */
/* Purpose:  This function configures the Meter mode                         */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_Meterconfig (ViSession instrSession, ViInt16 input, ViInt16 measure, 
					ViInt16 coupling, ViBoolean autoRange, ViInt16 VoltRang, ViInt16 OhmRang, 
					ViInt16 CapaRang)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
    
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(input,1,2))
		return VI_ERROR_PARAMETER2;
	if (mxOX50x2_invalidViInt16Range(measure,0,8))
		return VI_ERROR_PARAMETER3;
	if (mxOX50x2_invalidViInt16Range(coupling,0,2))
		return VI_ERROR_PARAMETER4;
	if ((mxOX50x2_invalidViBooleanRange(autoRange))) 
		return VI_ERROR_PARAMETER5;
	if (mxOX50x2_invalidViInt16Range(VoltRang,0,4))
		return VI_ERROR_PARAMETER6;
	if (mxOX50x2_invalidViInt16Range(OhmRang,0,6))
		return VI_ERROR_PARAMETER7;
	if (mxOX50x2_invalidViInt16Range(CapaRang,0,6))
		return VI_ERROR_PARAMETER8;
		
	/*-Set the meter parameters----------------------------------------------*/
	Fmt (wrBuffer, "FUNC %s\r", CmdMultMeas[measure]);
	if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return mxOX50x2_status;
		
	if(measure == 0)
		{
		Fmt (wrBuffer, "INP%i:DMM:COUP %s\r", input, MulCoup[coupling]);
		if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
			return mxOX50x2_status;
		}

	// No ranging for Continuity, Diode and RPM
	if(measure != 5 && measure != 7 && measure != 8)
		{
		Fmt (wrBuffer, "RANG%i:AUTO %i[b2]\r", input, autoRange);
		if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
			return mxOX50x2_status;
		}
	
	if(!autoRange)
		{
		switch (measure)
			{
			case 0:
				Fmt (wrBuffer, "RANG%i:VOLT %s\r", input, CmdMultVolt[VoltRang]);
				if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
					return mxOX50x2_status;	
				break;

			case 4:
				Fmt (wrBuffer, "RANG:OHM %s\r", CmdMultOhm[OhmRang]);
				if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
					return mxOX50x2_status;	
				break;
	
			case 6:
				Fmt (wrBuffer, "RANG:CAPA %s\r", CmdMulCapa[CapaRang]);
				if ((mxOX50x2_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
					return mxOX50x2_status;	
				break;
			}
		}
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}	

/*===========================================================================*/
/* Function: Read MeterMeas                                                  */
/* Purpose:  This function reads the main measurement to the meter mode      */
/*===========================================================================*/
ViStatus _VI_FUNC mxOX50x2_readMeterMeas (ViSession instrSession, ViInt16 ref, ViPReal64 reading)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;
	ViByte mult;
	
	/*-Check input parameter ranges------------------------------------------*/
	if (mxOX50x2_invalidViInt16Range(ref,1,2) && ref != 5)
		return VI_ERROR_PARAMETER2;

	/*-Set measure parameter-------------------------------------------------*/
	if ((mxOX50x2_status = viPrintf (instrSession,"MEAS:DMM? INT%i\r", ref)) < 0)
		return mxOX50x2_status;
	if ((mxOX50x2_status = viScanf (instrSession, "%lf%c", reading, &mult)) < 0)
		return mxOX50x2_status;		

	switch (mult)
		{
		case 'G':  *reading *= 1e+9;  break;
		case 'M':  *reading *= 1e+6;  break;
		case 'k':  *reading *= 1e+3;  break;
		case 'm':  *reading *= 1e-3;  break;
		case '':  *reading *= 1e-6;  break;
		case 'p':  *reading *= 1e-12; break;
		}
		
	/*-Check instrument status & Return from function------------------------*/
	if ((mxOX50x2_status = mxOX50x2_instrStatus(instrSession)) < 0)
		return mxOX50x2_status;

	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  mxOX50x2_close (ViSession instrSession)
{

	mxOX50x2_instrRange instrPtr;
	ViSession rmSession;
	ViStatus  mxOX50x2_status = VI_SUCCESS;

	if (( mxOX50x2_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return  mxOX50x2_status;
	if (( mxOX50x2_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return  mxOX50x2_status;
    
	if(instrPtr != NULL)
		free (instrPtr);
    
	mxOX50x2_status = viClose (instrSession);
	viClose (rmSession);

	return  mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean  mxOX50x2_invalidViBooleanRange (ViBoolean val)
{
	return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mxOX50x2_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mxOX50x2_invalidViInt32Range  (ViInt32 val, ViInt32 min, ViInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mxOX50x2_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  mxOX50x2_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mxOX50x2_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  mxOX50x2_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Instrument Status                                               */
/* Purpose:  This function checks the instrument error status, by reading    */
/*           and processing instrument's event status register. An error     */
/*           is reported if bits 4 and 5 resp. EXE and CME are asserted.     */
/*===========================================================================*/
ViStatus  mxOX50x2_instrStatus (ViSession instrSession)
{
	ViUInt32 retCnt = 0;
	ViStatus  mxOX50x2_status = VI_SUCCESS;
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte status;
    
	/*- Command to query Event Status Register ------------------------------*/
	if (( mxOX50x2_status = viPrintf (instrSession,"*ESR?\r")) < 0)
		return  mxOX50x2_status;
        
	if (( mxOX50x2_status = viRead (instrSession, rdBuffer, 50, &retCnt)) < 0)
		return  mxOX50x2_status;

	/*- Scanning the response -----------------------------------------------*/
	if (Scan (rdBuffer, "%i[b1]", &status) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
    
	/*- Testing bits 4 and 5 ------------------------------------------------*/
	if (status & 0x30) 
		return VI_ERROR_mxOX50x2_ERROR;
    
	return  mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the  mxOX50x2_init function.  When*/
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from  mxOX50x2_init to    */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus  mxOX50x2_initCleanUp (ViSession openRMSession,ViPSession openInstrSession,
                    ViStatus currentStatus)
{
	viClose (*openInstrSession);
	viClose (openRMSession);
	*openInstrSession = VI_NULL;
    
	return currentStatus;
}

/*===========================================================================*/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the  mxOX50x2_reset operation and by the  */
/*            mxOX50x2_init function if the reset option has not been        */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/                                    
/*===========================================================================*/
ViStatus  mxOX50x2_defaultInstrSetup (ViSession instrSession)
{
	ViStatus  mxOX50x2_status = VI_SUCCESS;
	ViUInt32 retCnt = 0;
	mxOX50x2_instrRange instrPtr;
        
	/* Determine if the structure has been initialized for the current VISA  */
	/* Session and malloc if it has not.                                     */
	if ( mxOX50x2_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr))
		return  mxOX50x2_status;
    
	if (instrPtr == NULL) 
		instrPtr = malloc (sizeof (struct  mxOX50x2_statusDataRanges));
    
	instrPtr -> triggerMode = 0;
	instrPtr -> val2 = 0;
	instrPtr -> val3 = 0;
	Fmt (instrPtr -> instrDriverRevision, "%s<%s",  mxOX50x2_REVISION);
    
	if (( mxOX50x2_status = viSetAttribute (instrSession, VI_ATTR_USER_DATA, (ViUInt32)instrPtr)) < 0)
		return  mxOX50x2_status;                                       

	if (( mxOX50x2_status = viWrite (instrSession, "*CLS;:FORM INT;:FORM:DINT OFF\r", 30, &retCnt)) < 0)
		return  mxOX50x2_status;

	return  mxOX50x2_status;    
}

/*===========================================================================*/
/* Function: Make data readable                                              */
/* Purpose:  This function fills output parameters with readable data        */
/*===========================================================================*/
ViStatus mxOX50x2_makeDataReadable  (ViByte tabByte[], ViReal64 waveformArray[], 
						ViPInt32 numberOfSamples, ViPReal64 XStart, ViPReal64 XIncrement, ViPReal64 YIncrement)
{
	ViStatus mxOX50x2_status = VI_SUCCESS;
	ViInt32 i, index, start, type = 0;
	ViReal64 Offset; 

//#define OFFSET_ADC	393216		//0x60000 en hexa
#define OFFSET_ADC	524288		//0x80000 en hexa

	
	if ((start = FindPattern (tabByte, 0, 20, "#510036", 0, 0)) != -1)
		type = 1;
		
	if (type)
		{
		*XIncrement = cmdTimeBase[tabByte[11]] / 250.0;
		*YIncrement = cmdVolt[tabByte[31]] / 32768.0;
		Offset = OFFSET_ADC;
		*numberOfSamples = 2500;
		start += 43;
		}
	else
		{
		/*-Find begining of data file----------------------------------------*/    
		start = FindPattern (tabByte, 0, 200, "#5100", 0, 0);
		if (start == -1) 
			return VI_ERROR_mxOX50x2_INVALID_FILE_CONTENT;
		start += 7;
	    index = FindPattern (tabByte, 0, 80, "SCALe", 0, 0);
		Scan(tabByte + index + 6, "%f", XIncrement);

		index = FindPattern (tabByte, 0, 90, "SIZE", 0, 0);
		Scan(tabByte + index + 5, "%i[b4]", numberOfSamples);
		 
		index = FindPattern (tabByte, 0, 200, "OFFSet", 0, 0);
		Scan(tabByte + index + 7, "%f", &Offset);
		Offset = OFFSET_ADC;
	
		index = FindPattern (tabByte, 100, 200, "SCALe", 0, 0);
		Scan(tabByte + index + 6, "%f", YIncrement);
		}
		
	for (i=0; i<*numberOfSamples; i++)
		waveformArray[i] = (*(long*)(tabByte + start + i * 4) - Offset) * *YIncrement;
		
	return mxOX50x2_status;
}

/*===========================================================================*/
/* Function: Check File Name                                                 */
/* Purpose:  This function checks if the file name is properly edited        */		
/*===========================================================================*/
ViBoolean mxOX50x2_invalidFilename (ViString filename, ViString type)
{
	ViInt16 startIndex = 0;
    
	if (strlen(filename + startIndex) > 12) 
		return VI_TRUE;
	startIndex = FindPattern (filename, 0, -1, ".", 0, 1);
	if (CompareStrings (filename, startIndex, type, 0, 0) != 0)  
		return VI_TRUE; 
		
	return VI_FALSE;
}

