#include "gpib.h"
#include "formatio.h"
#include "gx1304.h"

#define GX1304_REVISION   "A1.0"

/*  = GX1304 Metrix =====================================================*/
/*  LabWindows CVI 3.1 Instrument Driver                                 */
/*  Original Release: 02/15/96                                           */
/*  By: Francis VALOTEAU                                                 */
/*     Ph : 50 33 62 62       Fax : 50 33 62 71                          */
/*  Originally written in C                                              */
/*  Modification History: None                                           */
/*  =====================================================================*/

/*= INSTRUMENT TABLE ======================================================*/
/* address array: contains the GPIB addresses of opened instruments.       */
/* bd array: contains the device descriptors returned by OpenDev.          */
/* instr_cnt: contains the number of instruments open of this model type.  */
/*=========================================================================*/
static short address[GX1304_MAX_INSTR + 1];
static short bd[GX1304_MAX_INSTR + 1];
static short instr_cnt;

/*= STATIC VARIABLES ======================================================*/
static char cmd[100];      /* buffer for GPIB I/O strings              */
static short  gx1304_err;    /* error variable for the instrument module */
static double mainvpp;     /* Vpp of main generator                    */
static double maindco;     /* DC offset of main generator              */
static double mainfrq;     /* Frequency of main generator              */
static double sumvpp;      /* SUM amplitude                            */
static double frqdev;      /* Frequency deviation (FM)                 */

/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
static char *mfunction[8 + 1];      /* Main Generator                      */
static char *xfunction[6 + 1];      /* Auxiliary  Generator                */
static char *amfunction[4 + 1];     /* Amplitude Modulation                */
static char *sumfunction[4 + 1];    /* SUM                                 */
static char *vcafunction[4 + 1];    /* Voltage Controlled Amplitude        */
static char *fmfunction[4 + 1];     /* Frequency Modulation                */
static char *vcffunction[4 + 1];    /* Voltage Controlled Frequency        */
static char *stfunction[1 + 1];     /* Store / Recall                      */

/*= UTILITY ROUTINES ======================================================*/
short gx1304_open_instr (short, short *);
short gx1304_close_instr (short);
short gx1304_invalid_integer_range (short, short, short, short);
short gx1304_invalid_longint_range (long, long, long, short);
short gx1304_invalid_real_range (double , double , double , short);
short gx1304_device_closed (short);
short gx1304_read_data (short, char *, short);
short gx1304_write_data (short , char *, short);
short gx1304_read_data_file (short, char *);
short gx1304_write_data_file (short, char *);
short gx1304_poll (short, char *);
short gx1304_set_timeout (short , short, short *);
void gx1304_setup_arrays (void);

/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument and initializes the        */
/*           instrument to a known state.                                  */
/*=========================================================================*/
short gx1304_init (short addr,short id_query, short rest, short *instrID)
{
    short ID =0;

    if (gx1304_invalid_integer_range (addr, 0, 30,  -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (id_query, 0, 1, -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (rest, 0, 1, -3) != 0)
        return gx1304_err;

    if (gx1304_open_instr (addr, &ID) != 0)
        return gx1304_err;

    /* Identify Instrument */
    if (id_query) {
        if (gx1304_write_data (ID, "*IDN?", 5) != 0) {
            gx1304_close_instr (ID);
            return gx1304_err;
        }
        if (gx1304_read_data (ID, cmd, 50) != 0) {
            gx1304_close_instr (ID);
            return gx1304_err;
        }
        if (FindPattern (cmd, 0, -1, "1304", 0, 0)==-1) {
            gx1304_err = 223;
            gx1304_close_instr (ID);
            return gx1304_err;
        }
    }
    /* Reset Instrument */
    if (rest) {
        if (gx1304_write_data (ID, "*RST", 4) != 0) {
            gx1304_close_instr (ID);
            return gx1304_err;
        }
    }
    /* Initialize arrays */
    gx1304_setup_arrays ();
    *instrID = ID;

    /* Initialize statics */
    mainvpp = 1.0;
    maindco = 0.0;
    mainfrq = 10000.0;
    frqdev = 0.0;
    sumvpp = 0.0;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: MAIN Generator                                                */
/* Purpose:  Sets the following parameters; function, frequency, amplitude,*/
/*           symmetry, offset and phase.                                   */
/*=========================================================================*/
short gx1304_maing (short instrID, short mfunct, double mfrq, double mvpp, double msym, double mdco, double mphase)
{
    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (mfunct, 0, 8,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (mfrq, 0.01, 13000000.0,  -3) != 0)
        return gx1304_err;
    /* if +/- puls, 0.001 < mvpp < 10.0  */
    if (mfunct>6) {
        if (gx1304_invalid_real_range (mvpp, 0.001, 10.0,  -4) != 0)
            return gx1304_err;
    }
    else {
        if (gx1304_invalid_real_range (mvpp, 0.002, 20.0,  -4) != 0)
            return gx1304_err;
    }
    if (gx1304_invalid_real_range (msym, 5.0, 95.0,  -5) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (mdco, -10.0, 10.0,  -6) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (mphase, -360.0, 360.0,  -7) != 0)
        return gx1304_err;

    mainvpp = mvpp;
    mainfrq = mfrq;
    maindco = mdco;
    if (sumvpp / 2.0 + mainvpp / 2.0 + maindco > 10.0)  {
        gx1304_err =  -301;
        return gx1304_err;
    }
    if (mainfrq <= frqdev)  {
        gx1304_err =  -302;
        return gx1304_err;
    }
    if (mainfrq + frqdev > 13000000.0)  {
        gx1304_err =  -303;
        return gx1304_err;
    }
    Fmt (cmd, "%s<MSYM %f", msym);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    Fmt (cmd, "%s<MDCO %f", mdco);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    Fmt (cmd, "%s<MPHZ %f", mphase);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;
    gx1304_err = 0;

    Fmt (cmd, "%s<MAIN %s;MFRQ %f;MVPP %f;", mfunction[mfunct], mfrq, mvpp);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: AUXiliary Generator                                           */
/* Purpose:  Sets the following parameters; function, frequency, amplitude */
/*           and symmetry.                                                 */
/*=========================================================================*/
short gx1304_auxg (short instrID, short xfunct, double xfrq, double xvpp, double xsym)
{
    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (xfunct, 0, 6,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (xfrq, 0.005, 50000.0,  -3) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (xvpp, 0.002, 20.0,  -4) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (xsym, 5.0, 95.0,  -5) != 0)
        return gx1304_err;

    Fmt (cmd, "%s<XSYM %f", xsym);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    Fmt (cmd, "%s<AUX %s;XFRQ %f;XVPP %f;", xfunction[xfunct], xfrq, xvpp);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Amplitude Modulation                                          */
/* Purpose:  Sets the following parameter; function, modulation frequency  */
/*           and depth.                                                    */
/*=========================================================================*/
short gx1304_am (short instrID, short amfunct, double amfrq, double amdepth)
{
    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (amfunct, 0, 4,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (amfrq, 0.005, 50000.0,  -3) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (amdepth, 0.0, 99.0,  -4) != 0)
        return gx1304_err;

    Fmt (cmd, "%s<XFRQ %f;AMDP %f;AM %s;", amfrq, amdepth, amfunction[amfunct]);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    return gx1304_err;
}
/*=========================================================================*/
/* Function: Voltage Controlled Amplitude                                  */
/* Purpose:  Sets the following parameters; function, start amplitude,     */
/*           stop amplitude, sweep time.                                   */
/*=========================================================================*/
short gx1304_vca (short instrID, short vcafunct, double vcastart, double vcastop, double vcasweept)
{
    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (vcafunct, 0, 4,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (vcastart, 0.002, 20.0,  -3) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (vcastop, 0.002, 20.0,  -4) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (vcasweept, 0.00002, 200.0,  -5) != 0)
        return gx1304_err;

    /* Instrument Specific Error */
    if (vcastop / vcastart > 100.0)  {
        gx1304_err =  301;
        return gx1304_err;
    }
    Fmt (cmd, "%s<VCA %s;XPER %f;VSTA %f;VSTP %f;", vcafunction[vcafunct], vcasweept, vcastart, vcastop);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    return gx1304_err;
}
/*=========================================================================*/
/* Function: SUM                                                           */
/* Purpose:  Sets the following parameters; function, frequency, amplitude.*/
/*=========================================================================*/
short gx1304_sum (short instrID, short sumfunct, double sumfrq, double sumampl)
{
    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (sumfunct, 0, 4,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (sumfrq, 0.005, 50000.0,  -3) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (sumampl, 0.002, 20.0,  -4) != 0)
        return gx1304_err;
    sumvpp = sumampl;

    /* Instrument Specific Error */
    if (sumvpp / 2.0 + mainvpp / 2.0 + maindco > 10.0)  {
        gx1304_err =  301;
        return gx1304_err;
    }

    Fmt (cmd, "%s<SUM %s;XFRQ %f;SVPP %f;", sumfunction[sumfunct], sumfrq, sumampl);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    return gx1304_err;
}
/*=========================================================================*/
/* Function: Frequency Modulation                                          */
/* Purpose:  Sets the following parameters; function, frequency modulation,*/
/*           deviation.                                                    */
/*=========================================================================*/
short gx1304_fm (short instrID, short fmfunct, double fmfrq, double fmdev)
{
    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (fmfunct, 0, 4,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (fmfrq, 0.005, 50000.0,  -3) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (fmdev, 0.001, 13000000.0,  -4) != 0)
        return gx1304_err;
    frqdev = fmdev;

    /* Instrument Specific Error */
    if (mainfrq <= frqdev)  {
        gx1304_err =  301;
        return gx1304_err;
    }
    if (mainfrq + frqdev > 13000000.0)  {
        gx1304_err =  302;
        return gx1304_err;
    }

    Fmt (cmd, "%s<FM ON;XFRQ %f;FMDV %f;FM %s;", fmfrq, fmdev, fmfunction[fmfunct]);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    return gx1304_err;
}
/*=========================================================================*/
/* Function: Voltage Controlled Frequency                                  */
/* Purpose:  Sets the following parameters; function, start frequency,     */
/*           stop frequency, sweep time.                                   */
/*=========================================================================*/
short gx1304_vcf (short instrID, short vcffunct, double vcfstart, double vcfstop, double vcfsweept)
{
    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (vcffunct, 0, 4,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (vcfstart, 0.001, 13000000.0,  -3) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (vcfstop, 0.001, 13000000.0,  -4) != 0)
        return gx1304_err;
    if (gx1304_invalid_real_range (vcfsweept, 0.00002, 200.0,  -5) != 0)
        return gx1304_err;

    /* Instrument Specific Error */
    if (vcfstop / vcfstart > 10000.0)  {
        gx1304_err =  301;
        return gx1304_err;
    }
    Fmt (cmd, "%s<VCF %s;XPER %f;FSTA %f;FSTP %f;", vcffunction[vcffunct], vcfsweept, vcfstart, vcfstop);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    return gx1304_err;
}
/*=========================================================================*/
/* Function: STORE-RECALL                                                  */
/* Purpose:  Saves or recalls the complete instrument setup into or from   */
/*           the number given.                                             */
/*=========================================================================*/
short gx1304_storecall (short instrID, short stfunct, short stnumber)
{
    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (stfunct, 0, 1,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (stnumber, 1, 50,  -3) != 0)
        return gx1304_err;
    Fmt (cmd, "%s<%s %d;", stfunction[stfunct], stnumber);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1304_err;

    return gx1304_err;
}
/*=========================================================================*/
/* Function: TRANSFER                                                      */
/* Purpose:  transfers to and from the controller in the form of           */
/*           checksummed binary blocks.                                    */
/*=========================================================================*/
short gx1304_transfer (short instrID, short trffunct, short trfnumber, char *trffilename)
{
    short  handle;
    long size;

    if (gx1304_device_closed (instrID) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (trffunct, 0, 1,  -2) != 0)
        return gx1304_err;
    if (gx1304_invalid_integer_range (trfnumber, 1, 50,  -3) != 0)
        return gx1304_err;

    if (!trffunct)  {
        Fmt (cmd, "%s<STORE? %d", trfnumber);
        if (gx1304_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return gx1304_err;
        if (gx1304_read_data_file(instrID, trffilename) != 0)
            return gx1304_err;
    }
    else  {
        handle=OpenFile (trffilename, 0, 2, 1);
        if (handle==-1) {
            gx1304_err =  301;
            return gx1304_err;
        }
        if ((GetFileInfo (trffilename, &size))!=1) {
            gx1304_err =  301;
            return gx1304_err;
        }
        if ((size != 387L) && (size != 388L)) {
            gx1304_err =  302;
            return gx1304_err;
        }
        CloseFile (handle);
        if (gx1304_write_data_file(instrID, trffilename) != 0)
            return gx1304_err;
    }
    return gx1304_err;
}
/*=========================================================================*/
/* Function: Write To Instrument                                           */
/* Purpose:  This function writes a command string to the instrument.      */
/*=========================================================================*/
short gx1304_write (short instrID, char *cmd_string)
{
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_device_closed(instrID) != 0)
        return gx1304_err;

    Fmt (cmd, "%s<%s", cmd_string);
    if (gx1304_write_data (instrID, cmd, NumFmtdBytes()) != 0)
        return gx1304_err;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Read Instrument Buffer                                        */
/* Purpose:  This function reads the output buffer of the instrument.      */
/*=========================================================================*/
short gx1304_read (short instrID, short numbytes, char *in_buff, short *bytes_read)
{
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_device_closed(instrID) != 0)
        return gx1304_err;

    *bytes_read = 0;
    if (gx1304_read_data (instrID, in_buff, numbytes) != 0)
        return gx1304_err;

    *bytes_read = ibcnt;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Reset                                                         */
/* Purpose:  This function resets the instrument.                          */
/*=========================================================================*/
short gx1304_reset (short instrID)
{
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_device_closed(instrID) != 0)
        return gx1304_err;

    if (gx1304_write_data (instrID, "*RST", 4) != 0)
        return gx1304_err;

    return gx1304_err;
}
/*=========================================================================*/
/* Function: Self-Test                                                     */
/* Purpose:  This function executes the instrument self-test and returns   */
/* the result.                                                             */
/*=========================================================================*/
short gx1304_self_test (short instrID, short *test_result, char* test_message)
{
    short done,error;
	char  status;
	
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_device_closed(instrID) != 0)
        return gx1304_err;

    if (gx1304_set_timeout (instrID, 14, &error) != 0)
        return gx1304_err;
    if (gx1304_write_data (instrID, "*TST?", 5) != 0)
        return gx1304_err;
    done = 0;
    while (!done)  {
        if (gx1304_poll (instrID, &status) != 0)
            return gx1304_err;
        if ((status & 0x10) == 0x10)
            done = 1;
        if ((ibsta & (short)0x8000) == (int)0x8000)  {
            gx1304_err = 226;
            return gx1304_err;
        }
        if ((ibsta & 0x4000) == 0x4000)  {
            gx1304_err = 240;
            return gx1304_err;
        }
    }
    if (gx1304_set_timeout (instrID, 13, &error) != 0)
        return gx1304_err;
    if (gx1304_read_data (instrID, test_message, 2) != 0)
        return gx1304_err;
    if (Scan (test_message, "%s>%c", test_result) != 1)  {
        gx1304_err = 236;
        return gx1304_err;
        }
    if (*test_result=='0')
        Fmt (test_message,"%s<%s","Success!");
    else
        Fmt (test_message,"%s<%s","Failed!");

    return gx1304_err;
}
/*=========================================================================*/
/* Function: Revision                                                      */
/* Purpose:  This function returns the driver and instrument revisions.    */
/*=========================================================================*/
short gx1304_revision_query (short instrID, char *driver_rev, char *instr_rev)
{
    short index;

    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;
    if (gx1304_device_closed(instrID) != 0)
        return gx1304_err;

    if (gx1304_write_data (instrID, "*IDN?", 5) != 0)
        return gx1304_err;
    if (gx1304_read_data (instrID, cmd, 50) != 0)
        return gx1304_err;

    index=FindPattern(cmd, 0, -1, "1304", 0, 0);
    if (index==-1) {
        gx1304_err = 236;
        return gx1304_err;
    }
    else {
         index=FindPattern(cmd, index, -1, ".", 0, 0);
         if (index==-1) {
            gx1304_err = 236;
            return gx1304_err;
         }
         else {
            index--;
            CopyBytes (instr_rev, 0, cmd, index, 4);
         }
    }
    Fmt (driver_rev, "%s<%s", GX1304_REVISION);

    return gx1304_err;
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
short gx1304_close (short instrID)
{
    if (gx1304_invalid_integer_range (instrID, 1, GX1304_MAX_INSTR, -1) != 0)
        return gx1304_err;

    if (gx1304_device_closed (instrID))
        return gx1304_err;

    gx1304_close_instr (instrID);
    return gx1304_err;
}
/*= UTILITY ROUTINES ======================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table and the GPIB device table for the            */
/*           instrument.  The size of the Instrument Table can be changed  */
/*           in the include file by altering the constant                  */
/*           GX1304_MAX_INSTR.  The return value of this function is equal */
/*           to the global error variable.                                 */
/*=========================================================================*/
short gx1304_open_instr (short addr, short *ID)
{
    short i, instrID;

    instrID = 0;
    gx1304_err=0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= GX1304_MAX_INSTR; i++)
        if (address[i] == addr) {
            instrID = i;
            i = GX1304_MAX_INSTR;
         }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= GX1304_MAX_INSTR; i++)
            if (address[i] == 0) {
                instrID = i;
                i = GX1304_MAX_INSTR;
             }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0) {
        gx1304_err = 220;
        return gx1304_err;
     }

/*  If the device has not been opened in the GPIB device table (bd[ID] = 0),*/
/*  then open it.                                                           */

    if (bd[instrID] <= 0)    {
        if (instr_cnt <= 0)
            CloseInstrDevs("gx1304");
        bd[instrID] = OpenDev ("", "gx1304");
        if (bd[instrID] <= 0)    {
            gx1304_err = 220;
            return gx1304_err;
        }
        instr_cnt += 1;
        address[instrID] = addr;
     }

/*  Change the primary address of the device    */

    if (ibpad (bd[instrID], addr) < 0)   {
        gx1304_err = 233;
        return gx1304_err;
    }
    *ID = instrID;
    return gx1304_err;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
short gx1304_close_instr (short instrID)
{
    if (bd[instrID] != 0) {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt -= 1;
     }
    else
        gx1304_err = 221;

    return gx1304_err;
 }

/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
short gx1304_invalid_integer_range (short val, short min, short max, short err_code)
{
  if ((val < min) || (val > max))
        gx1304_err = err_code;
  else
        gx1304_err = 0;
  return gx1304_err;
}
/*=========================================================================*/
/* Function: Invalid Long Integer Range                                    */
/* Purpose:  This function checks a long integer to see if it lies between */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
short gx1304_invalid_longint_range (long val, long min, long max, short err_code)
{
    if (val < min || val > max)
        gx1304_err = err_code;
    else
        gx1304_err = 0;

    return gx1304_err;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
short gx1304_invalid_real_range (double val,double min,double max,short err_code)
{
    if ((val < min) || (val > max))
        gx1304_err = err_code;
    else
        gx1304_err = 0;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, set the      */
/*           global error variable to 232, 0 otherwise. The return value   */
/*           is equal to the global error value.                           */
/*=========================================================================*/
short gx1304_device_closed (short instrID)
{
    if (bd[instrID] <= 0)
        gx1304_err = 232;
    else
        gx1304_err = 0;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
short gx1304_read_data (short instrID, char *buf, short cnt)
{
    if (ibrd (bd[instrID], buf, (long)cnt) & 0x8000)
        gx1304_err = 231;
    else
        gx1304_err = 0;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
short gx1304_write_data (short instrID, char *buf, short cnt)
{
    if (ibwrt (bd[instrID], buf, (long)cnt) & 0x8000)
        gx1304_err = 230;
    else
        gx1304_err = 0;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Read Data File                                                */
/* Purpose:  This function reads a buffer of data from the instrument and  */
/*           stores it to the file specified by "filename".  Filename must */
/*           either be a string, such as "C:\\lw\\instr\\file" or a pointer*/
/*           to such a string.  The return value is equal to the global    */
/*           error variable.                                               */
/*=========================================================================*/
short gx1304_read_data_file (short instrID, char *filename)
{
    if (ibrdf (bd[instrID], filename) & 0x8000)
        gx1304_err = 229;
    else
        gx1304_err = 0;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Write Data File                                               */
/* Purpose:  This function writes a buffer of data from the file specified */
/*           by "filename" to the instrument. Filename must either be a    */
/*           string, such as "C:\\lw\\instr\\file" or a pointer to such a  */
/*           string.  The return value is equal to the global error        */
/*           variable.                                                     */
/*=========================================================================*/
short gx1304_write_data_file (short instrID, char *filename)
{
    if (ibwrtf (bd[instrID], filename) & 0x8000)
        gx1304_err = 228;
    else
        gx1304_err = 0;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Serial Poll                                                   */
/* Purpose:  This function performs a serial poll on the instrument.       */
/*           The status byte of the instrument is placed in the response   */
/*           variable. The return value is equal to the global error       */
/*           variable.                                                     */
/*=========================================================================*/
short gx1304_poll (short instrID, char *response)
{
    if (ibrsp (bd[instrID], response) & 0x8000)
        gx1304_err = 226;
    else
        gx1304_err = 0;

    return gx1304_err;
}

/*=========================================================================*/
/* Function: Set Timeout                                                   */
/* Purpose:  This function changes or disables the timeout of the device.  */
/*           Refer to the LabWindows Standard Libraries Reference Manual   */
/*           for timeout codes. The return value is equal to the global    */
/*           error variable.                                               */
/*=========================================================================*/
short gx1304_set_timeout (short instrID, short tmo_code, short *old_timeout)
{
    if (ibtmo (bd[instrID], tmo_code) & 0x8000)
        gx1304_err = 239;
    else {
        *old_timeout = iberr;
        gx1304_err = 0;
    }
    return gx1304_err;
}
/*=========================================================================*/
/* Function: Setup Arrays                                                  */
/* Purpose:  This function is called by the init routine to initialize     */
/*           global arrays.                                                */
/*=========================================================================*/
void gx1304_setup_arrays ()
{

    mfunction[0] = "OFF";
    mfunction[1] = "ON;MAIN SIN";
    mfunction[2] = "ON;MAIN TRI";
    mfunction[3] = "ON;MAIN SQR";
    mfunction[4] = "ON;MAIN DCO";
    mfunction[5] = "ON;MAIN POSPK";
    mfunction[6] = "ON;MAIN NEGPK";
    mfunction[7] = "ON;MAIN POSPUL";
    mfunction[8] = "ON;MAIN NEGPUL";
    xfunction[0] = "OFF";
    xfunction[1] = "ON;AUX SIN";
    xfunction[2] = "ON;AUX TRI";
    xfunction[3] = "ON;AUX SQR";
    xfunction[4] = "ON;AUX RMP";
    xfunction[5] = "ON;AUX FCV";
    xfunction[6] = "ON;AUX ACV";
    amfunction[0] = "OFF";
    amfunction[1] = "ON;AM SIN";
    amfunction[2] = "ON;AM TRI";
    amfunction[3] = "ON;AM SQR";
    amfunction[4] = "ON;AM EXT";
    sumfunction[0] = "OFF";
    sumfunction[1] = "ON;SUM SIN";
    sumfunction[2] = "ON;SUM TRI";
    sumfunction[3] = "ON;SUM SQR";
    sumfunction[4] = "ON;SUM EXT";
    vcafunction[0] = "OFF";
    vcafunction[1] = "ON;VCA INTLIN";
    vcafunction[2] = "ON;VCA INTLOG";
    vcafunction[3] = "ON;VCA EXTLIN";
    vcafunction[4] = "ON;VCA EXTLOG";
    fmfunction[0] = "OFF";
    fmfunction[1] = "ON;FM SIN";
    fmfunction[2] = "ON;FM TRI";
    fmfunction[3] = "ON;FM SQR";
    fmfunction[4] = "ON;FM EXT";
    vcffunction[0] = "OFF";
    vcffunction[1] = "ON;VCF INTLIN";
    vcffunction[2] = "ON;VCF INTLOG";
    vcffunction[3] = "ON;VCF EXTLIN";
    vcffunction[4] = "ON;VCF EXTLOG";
    stfunction[0] = "*SAV";
    stfunction[1] = "*RCL";
}

/*=== THE END =============================================================*/

