#include "gpib.h"
#include "formatio.h"
#include "gx1010.h"

/*= Metrix GX1010 10MHz Function Generator =======================*/
/* LabWindows 2.0 Multi-Instrument Driver                                            */
/* Original Release: May 1995                                                       */
/* By: John Tothill, Thurlby Thandar Instruments Ltd.                                                                     */
/* Originally written in C                                                 */
/* Modification History: Metrix GX1010 version                                              */
/*=========================================================================*/

/*= INSTRUMENT TABLE ======================================================*/
/* address array: contains the GPIB addresses of opened instruments.       */
/* bd array: contains the device descriptors returned by OpenDev.          */
/* instr_cnt: contains the number of instruments open of this model type.  */
/* gx1010_err: the error variable for the instrument module                */
/*=========================================================================*/
static int address[gx1010_MAX_INSTR + 1];
static int bd[gx1010_MAX_INSTR + 1];
static int instr_cnt;
static int gx1010_err;

/*= INTERNAL DATA =========================================================*/
/* cmd is a buffer for GPIB I/O strings */
static char cmd[200];

/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
static char *function[10];
static char *out[2];
static char *sens[2];
static char *mod_enab[3];
static char *mod_wave[4];
static char *swp_mode[2];
static char *swp_law[2];
static char *src[3];
static char *hop_enab[2];
static char *set_status[3];
static char *set_store[2];
static char *get_status[6];
static char *arb_store[2];
/*=========================================================================*/

/*= UTILITY ROUTINES ======================================================*/
int gx1010_open_instr (int);
int gx1010_close_instr (int);
int gx1010_invalid_integer_range (int, int, int, int);
int gx1010_invalid_longint_range (long, long, long, int);
int gx1010_invalid_real_range (double, double, double, int);
int gx1010_device_closed (int);
int gx1010_read_data (int, char *, int);
int gx1010_write_data (int, char *, int);
int gx1010_read_data_file (int, char *);
int gx1010_write_data_file (int, char *);
void gx1010_setup_arrays (void);
/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument, queries the instrument    */
/*           for its ID, and initializes the instrument to a known state.  */
/*=========================================================================*/
int gx1010_init (addr, id_query, rest, instrID)
int addr;
int id_query;
int rest;
int *instrID;
{
int  ID;

    if (gx1010_invalid_integer_range (addr, 0, 30, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (id_query, 0, 1, -2) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (rest, 0, 1, -3) != 0)
        return gx1010_err;

    ID = gx1010_open_instr (addr);
    if (ID <= 0)
        return gx1010_err;

    if (id_query)
        {
        if (gx1010_write_data (ID, "*IDN?\012", 6) != 0)
            {
            gx1010_close_instr (ID);
            return gx1010_err;
            }
        if (gx1010_read_data (ID, cmd, 50) != 0)
            {
            gx1010_close_instr (ID);
            return gx1010_err;
            }

        Scan (cmd, "1010");
        if (NumFmtdBytes () != 4)
            {
            gx1010_err = 223;
            gx1010_close_instr (ID);
            return gx1010_err;
            }
        }

    if (rest)
        {
        if (gx1010_write_data (ID, "*RST\012", 5) != 0)
            {
            gx1010_close_instr (ID);
            return gx1010_err;
            }
         }

    if (gx1010_write_data (ID, "BEEPMODE ERROR\012", 15) != 0)
        return gx1010_err;

    gx1010_setup_arrays ();
    *instrID = ID;
    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_config_main                                            */
/* Purpose:  Configure main generator waveform, frequency, output level and*/
/*           output state                                                  */
/*=========================================================================*/
int gx1010_config_main (instrID, func, output, freq, level, sense)
int instrID;
int func;
int output;
double freq;
double level;
int sense;
{
char *ptr;
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (func, 0, 9, -2) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (output, 0, 1, -3) != 0)
        return gx1010_err;
    if (gx1010_invalid_real_range (freq, 1e-4, 10e6, -4) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (sense, 0, 1, -9) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    if (func == 9)  /* noise selected */
        ptr = "";
    else
        ptr = "NOISE OFF;";
    Fmt(cmd,"%sFREQ %f;%s;EMFPP %f;OUTPUT %s,OUTPUT %s;\012"
                      ,ptr,freq,function[func],level,sens[sense],out[output]);
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_main_offset                                            */
/* Purpose:  set the main generator dc offset                              */
/*=========================================================================*/
int gx1010_main_offset (instrID, offset)
int instrID;
double offset;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_real_range (offset, -10.0, 10.0, -5) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    Fmt (cmd, "DCOFFS %f\012", offset);
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_main_symmetry                                          */
/* Purpose:  Set the main generator symmetry                               */
/*=========================================================================*/
int gx1010_main_symmetry (instrID, symm)
int instrID;
double symm;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_real_range (symm, 0.1, 99.9, -6) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    Fmt (cmd, "SYMM %f\012", symm);
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_main_phase                                             */
/* Purpose:  Set the main generator phase                                  */
/*=========================================================================*/
int gx1010_main_phase (instrID, phase)
int instrID;
int phase;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (phase, -360, 360, -7) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    Fmt (cmd, "PHASE %d", phase);
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_main_zout                                              */
/* Purpose:  Set the main generator output impedance                       */
/*=========================================================================*/
int gx1010_main_zout (instrID, zout)
int instrID;
int zout;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (zout, 0, 1, -8) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    if (zout == 1)
        zout = 600;
    else
        zout = 50;
    Fmt (cmd, "ZOUT %d", zout);
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_config_am                                              */
/* Purpose:  Configure amplitude modulation                                */
/*=========================================================================*/
int gx1010_config_am (instrID, enable, wave, depth, freq)
int instrID;
int enable;
int wave;
int depth;
double freq;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (enable, 0, 2, -2) != 0)
        return gx1010_err;
    if (gx1010_device_closed (instrID))
        return gx1010_err;
    if (enable == 2)  /* full configuration */
        {
        if (gx1010_invalid_integer_range (wave, 0, 2, -3) != 0)
            return gx1010_err;
        if (gx1010_invalid_integer_range (depth, 0, 100, -4) != 0)
            return gx1010_err;

        switch (wave)
        {
        case 2: /* external */
            Fmt (cmd, "AMSRC EXT;AM %s\012",mod_enab[enable]);
            break;
        case 0: /* sine */
            Fmt (cmd, "AMSRC TGEN;AMWAVE %s;AMDEPTH %d;AM %s\012"
                                ,mod_wave[wave],depth,mod_enab[enable]);
            break;
        case 1: /* square */
            if (gx1010_invalid_real_range (freq, 5e-3, 50e3, -5) != 0)
                return gx1010_err;
            freq = 1/freq;
            Fmt (cmd, "AMSRC TGEN;AMWAVE %s;AMDEPTH %d;TGEN %f;AM %s\012"
                             ,mod_wave[wave],depth,freq,mod_enab[enable]);
            break;
        }
        }
    else /* off or on so send only enable */
        Fmt (cmd,"AM %s\012",mod_enab[enable]);

    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;
    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_config_sweep                                           */
/* Purpose:  Configure sweep                                               */
/*=========================================================================*/
int gx1010_config_sweep (instrID,enable,swmode,law,time,startf,stopf
                                ,trig,mkra,mkrb)
int instrID;
int enable;
int swmode;
int law;
double time;
double startf;
double stopf;
int trig;
double mkra;
double mkrb;
{
char cmd2[100];
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (enable, 0, 2, -2) != 0)
        return gx1010_err;
    if (gx1010_device_closed (instrID))
        return gx1010_err;
    if (enable == 2)  /* full configuration */
        {
        if (gx1010_invalid_integer_range (swmode, 0, 1, -3) != 0)
            return gx1010_err;
        if (gx1010_invalid_integer_range (law, 0, 1, -4) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (time, 10e-3, 999, -5) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (startf, 1e-4, 10e6, -6) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (stopf, 1e-4, 10e6, -7) != 0)
            return gx1010_err;
        if (gx1010_invalid_integer_range (trig, 0, 2, -8) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (mkra, 1e-4, 10e6, -9) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (mkrb, 1e-4, 10e6, -10) != 0)
            return gx1010_err;

        Fmt (cmd2, "SWEEP OFF;SWPBEGFRQ %f;SWPENDFRQ %f;SWPMKRFRQ %f;"
                                                    ,startf,stopf,mkra);
        Fmt (cmd,"%sFREQ %f;SWPMODE %s;SWPLAW %s;SWPTIME %f;SWPSRC %s;SWEEP ON\012"
                     ,cmd2,mkrb,swp_mode[swmode],swp_law[law],time,src[trig]);
        }
    else /* off or on so send only enable */
        Fmt (cmd,"SWEEP %s\012",mod_enab[enable]);

    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;
    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_fsk                                                    */
/* Purpose:  Configure Frequency Shift Keying                              */
/*=========================================================================*/
int gx1010_fsk (instrID, enable, trig, freqa, freqb, trgfreq)
int instrID;
int enable;
int trig;
double freqa;
double freqb;
double trgfreq;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (enable, 0, 2, -2) != 0)
        return gx1010_err;
    if (gx1010_device_closed (instrID))
        return gx1010_err;
    if (enable == 2)  /* full configuration */
        {
        if (gx1010_invalid_integer_range (trig, 0, 2, -3) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (freqa, 1e-4, 10e6, -4) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (freqb, 1e-4, 10e6, -5) != 0)
            return gx1010_err;

        switch (trig)
        {
        case 2: /* external */
            Fmt (cmd, "FSKSRC EXT;FSKFRQA %f;FSKFRQB %f;FSK %s\012"
                                ,freqa,freqb,mod_enab[enable]);
            break;
        case 1: /* man.remote */
            Fmt (cmd, "FSKSRC MAN;FSKFRQA %f;FSKFRQB %f;FSK %s\012"
                                ,freqa,freqb,mod_enab[enable]);
            break;
        case 0: /* tgen */
            if (gx1010_invalid_real_range (trgfreq, 5e-3, 50e3, -6) != 0)
                return gx1010_err;
            trgfreq = 1/trgfreq;
            Fmt (cmd, "FSKSRC TGEN;FSKFRQA %f;FSKFRQB %f;TGEN %f;FSK %s\012"
                             ,freqa,freqb,trgfreq,mod_enab[enable]);
            break;
        }
        }
    else /* off or on so send only enable */
        Fmt (cmd,"FSK %s\012",mod_enab[enable]);

    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;
    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_hop                                                    */
/* Purpose:  Run or Configure Hop mode                                     */
/*=========================================================================*/
int gx1010_hop (instrID, enable, step, func, freq, level, offset, last,time)
int instrID;
int enable;
int step;
int func;
double freq;
double level;
double offset;
int last;
double time;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (enable, 0, 2, -2) != 0)
        return gx1010_err;
    if (gx1010_device_closed (instrID))
        return gx1010_err;
    if (enable == 2)  /* step configuration */
        {
        if (gx1010_invalid_integer_range (step, 0, 15, -3) != 0)
            return gx1010_err;
        if (gx1010_invalid_integer_range (func, 0, 8, -8) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (freq, 1e-4, 10e6, -4) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (offset, -10, 10, -5) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (time, 0, 65.0, -7) != 0)
            return gx1010_err;

        Fmt (cmd, "HOP OFF,0;SETHOP %d,%f,%f,%f,%s,%f\012"
                            ,step,time,freq,level,function[func],offset );
        }
    else /* off or on so send only enable */
        {
        if (gx1010_invalid_integer_range (last, 0, 15, -6) != 0)
            return gx1010_err;
        Fmt (cmd,"HOP %s,%d\012",hop_enab[enable],last);
        }

    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;
    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_trig                                                   */
/* Purpose:  Gives access to the instrumnets trigger capabilities.         */
/*=========================================================================*/
int gx1010_trig (instrID, enab, srce, bcnt, tgen, phase)
int instrID;
int enab;
int srce;
double bcnt;
double tgen;
int phase;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (enab, 0, 2, -2) != 0)
        return gx1010_err;
    if (gx1010_device_closed (instrID))
        return gx1010_err;
    if (enab == 2)  /* full configuration */
        {
        if (gx1010_invalid_integer_range (srce, 0, 2, -3) != 0)
            return gx1010_err;
        if (gx1010_invalid_real_range (bcnt, 0.5, 1023, -4) != 0)
            return gx1010_err;
        if (gx1010_invalid_integer_range (phase, -360, +360, -6) != 0)
            return gx1010_err;

        switch (srce)
        {
        case 2: /* external */
            Fmt (cmd, "TRIG OFF;TRIGSRC EXT;BCNT %f;PHASE %d;TRIG %s\012"
                                ,bcnt,phase,mod_enab[enab]);
            break;
        case 1: /* man.remote */
            Fmt (cmd, "TRIG OFF;TRIGSRC MAN;BCNT %f;PHASE %d;TRIG %s\012"
                                ,bcnt,phase,mod_enab[enab]);
            break;
        case 0: /* tgen */
            if (gx1010_invalid_real_range (tgen, 5e-3, 50e3, -5) != 0)
                return gx1010_err;
            tgen = 1/tgen;
            Fmt (cmd, "TRIG OFF;TRIGSRC TGEN;TGEN %f;BCNT %f;PHASE %d;TRIG %s\012"
                                ,tgen,bcnt,phase,mod_enab[enab]);
            break;
        }
        }
    else /* off or on so send only enable */
        Fmt (cmd,"TRIG %s\012",mod_enab[enab]);

    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;
    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_gate                                                   */
/* Purpose:  Gives access to the instrumnets gating capabilities.         */
/*=========================================================================*/
int gx1010_gate (instrID, enab, srce, tgen)
int instrID;
int enab;
int srce;
double tgen;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (enab, 0, 2, -2) != 0)
        return gx1010_err;
    if (gx1010_device_closed (instrID))
        return gx1010_err;
    if (enab == 2)  /* full configuration */
        {
        if (gx1010_invalid_integer_range (srce, 0, 2, -3) != 0)
            return gx1010_err;

        switch (srce)
        {
        case 2: /* external */
            Fmt (cmd, "GATE OFF;GATESRC EXT;GATE %s\012"
                                ,mod_enab[enab]);
            break;
        case 1: /* man.remote */
            Fmt (cmd, "GATE OFF;GATESRC MAN;GATE %s\012"
                                ,mod_enab[enab]);
            break;
        case 0: /* tgen */
            if (gx1010_invalid_real_range (tgen, 5e-3, 50e3, -4) != 0)
                return gx1010_err;
            tgen = 1/tgen;
            Fmt (cmd, "GATE OFF;GATESRC TGEN;TGEN %f;GATE %s\012"
                                ,tgen,mod_enab[enab]);
            break;
        }
        }
    else /* off or on so send only enable */
        Fmt (cmd,"GATE %s\012",mod_enab[enab]);

    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;
    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_set_status                                             */
/* Purpose:  Set the value in the Service Request Enable, Event Status     */
/*           Enable or Parallel Poll Enable register                       */
/*=========================================================================*/
int gx1010_set_status (instrID, reg, val)
int instrID;
int reg;
int val;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (reg, 0, 2, -2) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (val, 0, 255, -3) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    Fmt (cmd, "%s %d\012", set_status[reg],val);
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_send_trg                                               */
/* Purpose:  Send a trigger command to the GX1010. If MAN/REMOTE is        */
/*           selected as the trigger source for any active mode then that  */
/*           mode will be triggered by this command. If no modes are active*/
/*           then the command will be ignored.                             */
/*=========================================================================*/
int gx1010_send_trg (instrID)
int instrID;
{
    if (gx1010_device_closed (instrID))
        return gx1010_err;

    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;

    Fmt(cmd,"*TRG\012");
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;
    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_loc_stores                                             */
/* Purpose:  Allows saving to or recalling from the instruments local non- */
/*           volatile stores                                               */
/*=========================================================================*/
int gx1010_loc_stores (instrID, op, storeno)
int instrID;
int op;
int storeno;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (op, 0, 1, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (storeno, 0, 9, -2) != 0)
        return gx1010_err;
    if (op == 1 && storeno == 0)
        {
        gx1010_err = -2; /* can't write to store 0 */
        return gx1010_err;
        }
    if (gx1010_device_closed (instrID))
        return gx1010_err;

    Fmt (cmd, "%s %d\012", set_store[op],storeno);
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_get status                                             */
/* Purpose:  Reads the value in the selected iEEE 488.2 status register    */
/*=========================================================================*/
int gx1010_get_status (instrID, reg, val)
int instrID;
int reg;
int *val;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (reg, 0, 5, -2) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    Fmt (cmd, "%s\012", get_status[reg]);
    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;
    if (gx1010_read_data (instrID, cmd, 50) != 0)
        return gx1010_err;
    if (Scan (cmd, "%d",val) != 1)
        gx1010_err = 236;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_snd_cmd                                                */
/* Purpose:  Sends the data in the supplied buffer to the instrument       */
/*           Use for commands not available in this driver - e.g. inc and  */
/*           dec commands.												   */
/*=========================================================================*/
int gx1010_snd_cmd (instrID, command)
int instrID;
char *command;
{
char output_buf[200];
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    Fmt(output_buf,"%s",command);
    if (gx1010_write_data(instrID,command,NumFmtdBytes()) != 0)
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_get_response                                           */
/* Purpose:  Gets a response to a previous query. Used in conjuction with  */
/*           the send command function                                     */
/*=========================================================================*/
int gx1010_get_resp (instrID, response, nbytes)
int instrID;
char *response;
int *nbytes;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    if (gx1010_read_data(instrID,response,2000))
        return gx1010_err;
    *nbytes = ibcnt;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_file_ops                                               */
/* Purpose:  Send or receive files                                         */
/*=========================================================================*/
int gx1010_file_ops (instrID, op, fname)
int instrID;
int op;
char *fname;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (op, 0, 1, -2) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    if (op == 1) /* get to file from gx1010 */
        {
        if (gx1010_read_data_file(instrID,fname))
            return gx1010_err;
        }
    else /* send from file to gx1010 */
        if (gx1010_write_data_file(instrID,fname))
            return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_new_arb                                                */
/* Purpose:  Download a new ARB or staircase waveform from disk           */
/*=========================================================================*/
int gx1010_new_arb (instrID, fname)
int instrID;
char *fname;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    if (gx1010_write_data_file(instrID,fname))
        return gx1010_err;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: gx1010_arb_stores                                             */
/* Purpose:  Allows saving to or recalling from the instruments local non- */
/*           volatile waveform stores                                               */
/*=========================================================================*/
int gx1010_arb_stores (instrID, op, storeno, name)
int instrID;
int op;
int storeno;
char *name;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;
    if (gx1010_invalid_integer_range (op, 0, 1, -1) != 0)
        return gx1010_err;
    if ((op == 1 && storeno > 5) || storeno < 1)
        {
        gx1010_err = -2; /* can't write to store 0 */
        return gx1010_err;
        }
    if (gx1010_device_closed (instrID))
        return gx1010_err;

    if (op == 1) /* save */
        Fmt (cmd, "%s %d,%s\012", arb_store[op],storeno,name);
    else
        Fmt (cmd, "%s %d\012", arb_store[op],storeno);

    if (gx1010_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return gx1010_err;

    return gx1010_err;
}


/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int gx1010_close (instrID)
int instrID;
{
    if (gx1010_invalid_integer_range (instrID, 1, gx1010_MAX_INSTR, -1) != 0)
        return gx1010_err;

    if (gx1010_device_closed (instrID))
        return gx1010_err;

    gx1010_close_instr (instrID);
    return gx1010_err;
}

/*= UTILITY ROUTINES ======================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table and the GPIB device table for the            */
/*           instrument.  If successful, the instrument ID is returned,    */
/*           else a -1 is returned.  The size of the Instrument Table can  */
/*           be changed in the include file by altering the constant       */
/*           gx1010_MAX_INSTR.                                             */
/*=========================================================================*/
int gx1010_open_instr (addr)
int addr;
{
    int i, instrID;

    instrID = 0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= gx1010_MAX_INSTR; i++)
        if (address[i] == addr)
            {
            instrID = i;
            i = gx1010_MAX_INSTR;
            }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= gx1010_MAX_INSTR; i++)
            if (address[i] == 0)
                {
                instrID = i;
                i = gx1010_MAX_INSTR;
                }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0)
        {
        gx1010_err = 220;
        return -1;
        }

/*  If the device has not been opened in the GPIB device table (bd[ID] = 0),*/
/*  then open it.                                                           */

    if (bd[instrID] <= 0)
        {
        if (instr_cnt <= 0)
            CloseInstrDevs("gx1010");
        bd[instrID] = OpenDev ("", "gx1010");
        if (bd[instrID] <= 0)
            {
            gx1010_err = 220;
            return -1;
            }
        instr_cnt += 1;
        address[instrID] = addr;
        }

/*  Change the primary address of the device    */

    if (ibpad (bd[instrID], addr) < 0)
        {
        gx1010_err = 233;
        return -1;
        }

    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int gx1010_close_instr (instrID)
int instrID;
{
    if (bd[instrID] != 0)
        {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt -= 1;
        }
    else
        gx1010_err = 221;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int gx1010_invalid_integer_range (val, min, max, err_code)
int val;
int min;
int max;
int err_code;
{
  if ((val < min) || (val > max))
    {
    gx1010_err = err_code;
    return -1;
    }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Long Integer Range                                    */
/* Purpose:  This function checks a long integer to see if it lies between */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int gx1010_invalid_longint_range (val, min, max, err_code)
long val;
long min;
long max;
int  err_code;
{
    if (val < min || val > max)
        {
        gx1010_err = err_code;
        return -1;
        }
    return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int gx1010_invalid_real_range (val, min, max, err_code)
double val;
double min;
double max;
int err_code;
{
  if ((val < min) || (val > max))
    {
    gx1010_err = err_code;
    return -1;
    }
  return 0;
}
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int gx1010_device_closed (instrID)
int instrID;
{
    if (bd[instrID] <= 0)
        {
        gx1010_err = 232;
        return -1;
        }
    return 0;
}
/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int gx1010_read_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibrd(bd[instrID], buf, (long)cnt) <= 0)
        gx1010_err = 231;
    else
        gx1010_err = 0;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int gx1010_write_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibwrt(bd[instrID], buf, (long)cnt) <= 0)
        gx1010_err = 230;
    else
        gx1010_err = 0;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: Read Data File                                                */
/* Purpose:  This function reads a buffer of data from the instrument and  */
/*           stores it to the file specified by "filename".  Filename must */
/*           either be a string, such as "C:\lw\instr\file" or a pointer   */
/*           to such a string.  The return value is equal to the global    */
/*           error variable.                                               */
/*=========================================================================*/
int gx1010_read_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibrdf (bd[instrID], filename) <= 0)
        gx1010_err = 229;
    else
        gx1010_err = 0;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: Write Data File                                               */
/* Purpose:  This function writes a buffer of data from the file specified */
/*           by "filename" to the instrument. Filename must either be a    */
/*           string, such as "C:\lw\instr\file" or a pointer to such a     */
/*           string.  The return value is equal to the global error        */
/*           variable.                                                     */
/*=========================================================================*/
int gx1010_write_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibwrtf (bd[instrID], filename) <= 0)
        gx1010_err = 228;
    else
        gx1010_err = 0;

    return gx1010_err;
}
/*=========================================================================*/
/* Function: Setup Arrays                                                  */
/* Purpose:  This function is called by the init routine to initialize     */
/*           global arrays.                                                */
/*=========================================================================*/
void gx1010_setup_arrays ()
{
    function[0] = "SINE";
    function[1] = "TRIAN";
    function[2] = "SQUARE";
    function[3] = "POSPUL";
    function[4] = "NEGPUL";
    function[5] = "POSRAMP";
    function[6] = "NEGRAMP";
    function[7] = "STAIR";
    function[8] = "ARB";
    function[9] = "NOISE ON";

    out[0] = "OFF";
    out[1] = "ON";

    sens[0] = "NORMAL";
    sens[1] = "INVERT";

    mod_enab[0] = "OFF";
    mod_enab[1] = "ON";
    mod_enab[2] = "ON";

    mod_wave[0] = "SINE";
    mod_wave[1] = "SQUARE";
    mod_wave[2] = "SQUARE";

    swp_mode[0] = "ETOB";
    swp_mode[1] = "BTOE";

    swp_law[0] = "LIN";
    swp_law[1] = "LOG";

    src[0] = "CONT";
    src[1] = "MAN";
    src[2] = "EXT";

    hop_enab[0] = "OFF";
    hop_enab[1] = "RUN";

    set_status[0] = "*SRE";
    set_status[1] = "*ESE";
    set_status[2] = "*PRE";

    set_store[0] = "*RCL";
    set_store[1] = "*SAV";

    arb_store[0] = "ARBRCL";
    arb_store[1] = "ARBSAV";

    get_status[0] = "*STB?";
    get_status[1] = "*ESR?";
    get_status[2] = "*SRE?";
    get_status[3] = "*ESE?";
    get_status[4] = "*PRE?";
    get_status[5] = "EER?";
}
/*= THE END ===============================================================*/
