/*= ASYC II ===============================================================*/
/* LabWindows Instrument Driver for MX53, MX54, MX55, MX56.                */
/* Original Release: November ,15th,1995                                   */
/* By: G. COMMUNAL & F. VALOTEAU                                           */
/* Originally written in C                                                 */
/*                                                                         */
/* Modification History:                                                   */
/* By: O.LEMAIRE                                                           */
/* Release: November ,19th,1996                                            */
/* Benchtop integration (MX553, MX554, MX556)                              */
/*=========================================================================*/

#include <utility.h>
#include "rs232.h"
#include "formatio.h"
#include "asyc2.h"

/*= STATIC VARIABLES ======================================================*/
static int  port;                    /* number of the port opened                 */
static int  rscnt;                   /* number of bytes read or written on RS232  */
static int  asyc2_err;               /* error variable for the instrument module  */

static int  asyc2_instrument;        /* model of the instrument                   */
static int  asyc2_version;           /* version of the instrument                 */
static int  asyc2_ohm_value;         /* reference ohm                             */
static long asyc2_clock;             /* timer of the instrument                   */
static int  asyc2_ctrl_config;       /* configuration of the measurement          */
static int  asyc2_range;             /* range of the measurement                  */
static int  asyc2_ranging;           /* flag AUTO RANGING mode (manual/AUTO)      */
static int  asyc2_rel;               /* flag REL mode  (Off/Last/Ref)             */
static int  asyc2_mem;               /* flag MEM mode  (Off/On)                   */
static int  asyc2_lrhd;              /* flag LRHD mode (Off/On)                   */
static int  asyc2_surv;              /* flag SURV mode (Off/On)                   */
static int  asyc2_zoom;              /* flag ZOOM mode (Off/On)                   */
static int  asyc2_commut;            /* position of the rotary switch (0..15)     */
static int  asyc2_sbat;              /* level of the threshold of the battery     */
static int  asyc2_degre;             /* flag Temperature unit (degre C/degre F)   */
static int  asyc2_sensorpt;          /* flag Temperature sensor                   */
static int  asyc2_resol;             /* resolution of the instrument              */
static int  asyc2_reject;            /* flag Rejection ()                         */
static int  asyc2_Ze;                /* flag Input impedance  (GOHM/ 10 MOHM)     */
static int  asyc2_type;              /* flag type (Normal/Calibration)            */
static int  asyc2_batt;              /* flag Battery                              */
static int  asyc2_fuse1;             /* flag Fuse 1                               */
static int  asyc2_fuse2;             /* flag Fuse 2                               */
static char asyc2_sdate[15];         /* date of the last calibration (string)     */
static long asyc2_ldate;             /* date of the last calibration (long)       */
static char asyc2_smeasure[20];      /* measurement (string)                      */
static double asyc2_dmeasure;        /* measurement (double)                      */
static double asyc2_coef;            /* coef of the measurement                   */
static char asyc2_code_eeprom[10];   /* code read in the EEPROM                   */
static int  i;                       /* int  parameter for asyc2_status function call */
static long l;                       /* long parameter for asyc2_status function call */
static int  asyc2_benchtop;          /* flag benchtop                             */

/*= UTILITY ROUTINES ======================================================*/
int asyc2_invalid_short_range (short val, short min, short max, int err_code);
int asyc2_invalid_integer_range (int val, int min, int max, int err_code);
int asyc2_invalid_longint_range (long val, long min, long max, int err_code);
int asyc2_invalid_real_range (double val, double min, double max, int err_code);
int asyc2_device_closed (void);
int asyc_read_data (char *buf, int cnt, int term);
int asyc2_write_data (char *buf, int cnt);

int asyc2_send (char *buf, int cnt); /* write data and wait ACK form ASYC II */
int asyc2_decode(void);              /* Format measurement (string & double) */
int asyc2_change_range (int range);  /* change the range of measurement      */
int asyc2_change_config (int relat,int sur,int memo,int lrh,int zoo);
int asyc2_read_eeprom(int adresse);  /* read eeprom of multimeter            */
int asyc2_read_data(char *,int,int); /* read data on serial port */

/*=========================================================================*/
/* Function: init                                                          */
/* Purpose:  This function opens a com port for the instrument module,     */
/*           and reads the setup of the multimeter (eeprom).               */
/*=========================================================================*/
int asyc2_init(int com_port)
{
/*  check for invalid arguments */
    if (asyc2_invalid_integer_range (com_port, 1, 16, -1) != 0)
       return asyc2_err;

/*  Open a com port for the instrument module.  If an error occurs, set */
/*  asyc2_err = rs232err+400                                            */
    OpenComConfig (com_port, "",2400,0,8,1,512,512);
    if(rs232err!=0){
       asyc2_err=rs232err+400;
       return asyc2_err;
    }

/*  Set the timeout value for the instrument. If an error occurs, set   */
/*  asyc2_err = 239 "Error in configuring timeout".                     */
    SetComTime(com_port,1.0);
    if(rs232err!=0){
       asyc2_err=239;
       return asyc2_err;
    }

/*  Set port to the number of the port just opened.                     */
    port = com_port;

/*  Read the reference of the instrument . If an error occurs,         */
/*  set asyc2_err = 223 "Instrument identification query failed".      */
    if (asyc2_read_eeprom(60)!=0)
        return asyc2_err;
    if ( (Scan(asyc2_code_eeprom,"%s>%x",&asyc2_instrument) !=1) ||
         ( (asyc2_invalid_integer_range (asyc2_instrument, 0,  5, 223) != 0) &&
           (asyc2_invalid_integer_range (asyc2_instrument, 8, 13, 223) != 0)) ){
       asyc2_err=223;
       return asyc2_err;
    }

    /* Detect if it's a benchtop or not                                 */
    asyc2_read_eeprom(59);
    if(asyc2_code_eeprom[0]=='A' && asyc2_code_eeprom[1]=='A')
        asyc2_benchtop=1;
    else asyc2_benchtop=0;

/*  verify the configuration of the asyc2 */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

    /* Read OHM reference and store into int asyc2_ohm_value */
    asyc2_read_eeprom(55);
    Scan(asyc2_code_eeprom,"%s>%i",&asyc2_ohm_value);

    /* Read Timer Printer and store into long asyc2_clock */
    asyc2_read_eeprom(53);
    Scan(asyc2_code_eeprom,"%s>%i[b4]",&asyc2_clock);
    asyc2_read_eeprom(54);
    asyc2_clock=asyc2_clock+10000*(long)(asyc2_code_eeprom[3]-0x30);

    /* Read Calibration Date and store into string asyc2_sdate */
    /* and into long asyc2_ldate                               */
    asyc2_read_eeprom(61);
    asyc2_sdate[0]=asyc2_code_eeprom[2];
    asyc2_sdate[1]=asyc2_code_eeprom[3];
    asyc2_read_eeprom(62);
    asyc2_sdate[2]=asyc2_code_eeprom[0];
    asyc2_sdate[3]=asyc2_code_eeprom[1];
    asyc2_sdate[4]=asyc2_code_eeprom[2];
    asyc2_sdate[5]=asyc2_code_eeprom[3];
    /* Format string asyc2_sdate to long asyc2_ldate */
    Scan (asyc2_sdate,"%s>%i[b4]",&asyc2_ldate);

    /* Set interval for repetitive measurement to 1 second */
    asyc2_timer("1");

    return asyc2_err;
}
/*=========================================================================*/
/* Function: Voltage                                                       */
/* Purpose:  This function switch the instrument to voltage measurement      */
/*=========================================================================*/
int asyc2_voltage(int config,int gam,int relat,double ref,int zoom,
      int memo,int lrh,int sur)
{
    char    s[20],val[20];
    int     range_signal;
    int     q;

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (config, 0, 4, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (gam, 0, 5, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (relat, 0, 2, -3) != 0)
       return asyc2_err;
    switch (gam)
        {
        case 0: /* AUTO or 1000V : +- 1000V */
        case 5:
            if (asyc2_invalid_real_range (ref, -1000.0, 1000.0, -4) != 0)
            return asyc2_err;
            break;
        case 1:   /* 500mV : +- 500mV */
            if (asyc2_invalid_real_range (ref, -0.5, 0.5, -4) != 0)
            return asyc2_err;
            break;
        case 2:   /* 5V : +- 5V  */
            if (asyc2_invalid_real_range (ref, -5, 5, -4) != 0)
            return asyc2_err;
            break;
        case 3:   /* 50V : +- 50V */
            if (asyc2_invalid_real_range (ref, -50, 50, -4) != 0)
            return asyc2_err;
            break;
        case 4:  /* 500V : +- 500V */
            if (asyc2_invalid_real_range (ref, -500, 500, -4) != 0)
            return asyc2_err;
            break;
        }
    if (asyc2_invalid_integer_range (zoom, 0, 1, -5) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -6) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (lrh, 0, 1, -7) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -8) != 0)
       return asyc2_err;

/*  Convert config into instrument parameter */
    if (config==1) config=4;
    else if (config==2) config=6;
    else if (config==3) config=2;
    else if (config==4) config=3;
    gam--;
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  ZOOM and SURV not available on MX53 and MX55 */
    if(((asyc2_instrument==MX53)||(asyc2_instrument==MX55))&&(zoom||sur)){
       asyc2_err=300;
       return asyc2_err;
    }
/*  Check compatibility of parameters             */
/*  Error 302 : Two arguments are incompatible    */
    if((ref<0 && relat<0 && (config==PEAKP || config==AC || config==ACDC)) ||
       (ref>0 && relat<0 && config==PEAKM) ||
       (sur!=0 && gam!=asyc2_range) ||
       ((config==PEAKP || config==PEAKM) && (sur!=0 || lrh!=0))||
       (memo && lrh)){
       asyc2_err=302;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch    */
/*  Error 301: Wrong position of the rotary switch */
    if(/* if 500mV asked and rotary switch is not MV */
       ((gam==0)^(asyc2_commut==MV)) ||
       /* if rotary switch is not (VDC or mV or VAC) */
       asyc2_commut<VAC || asyc2_commut>VDC ||
       /* if not AC asked and rotary switch is VAC */
       (config!=AC && asyc2_commut==VAC ) ||
       /* if AC asked and rotary switch is VDC  */
       (config==AC && asyc2_commut==VDC ) ||
       /* if AC asked and rotary switch is mV */
       (config==AC && asyc2_commut==MV)) {
       asyc2_err=301;
       return asyc2_err;
    }

/* If current Function is different than Function to be set */
/* Set the instrument to voltage measurement */
    if(asyc2_ctrl_config!=config+16){
       Fmt(s,"%s<21%i\r",config);
       if(asyc2_send(s,4)!=0) return asyc2_err;
    }

/* If current Function is VAC or (VDC and not (PEAK+ or PEAK-)) */
/* Set the range */
    if(asyc2_commut==VAC || (asyc2_commut==VDC && !(gam==AUTO &&
       (config==PEAKP || config==PEAKM))))
       asyc2_change_range(gam);

/*  Set the relative reference */
    if(relat<0 ) {
       if(ref>=-0.5        && ref<=0.5   && asyc2_commut==MV) range_signal=0;
       else if(ref>=-5.0   && ref<=5.0   && asyc2_commut!=MV) range_signal=1;
       else if(ref>=-50.0  && ref<=50.0  && asyc2_commut!=MV) range_signal=2;
       else if(ref>=-500.0 && ref<=500.0 && asyc2_commut!=MV) range_signal=3;
       else if(asyc2_commut!=MV)                              range_signal=4;
       q=range_signal;
       if(ref<0){
         range_signal=range_signal+8;
         ref=ref*(-1);
       }
       ref=ref*1E+05;
       Fmt(val,"%s<%f[w10p0]",ref);
       if(asyc2_resol!=0) Fmt(s,"%s<23:%c%c%c%c%c%c\r",
       val[9-q],val[8-q],val[7-q],val[6-q],val[5-q],48+range_signal);
       else Fmt(s,"%s<23:%c%c%c%c0%c\r",
       val[8-q],val[7-q],val[6-q],val[5-q],48+range_signal);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,lrh,zoom);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Current                                                       */
/* Purpose:  This function switch the instrument to current measurement      */
/*=========================================================================*/
int asyc2_current(int config,int gam,int relat,double ref,int zoom,
       int memo,int sur)
{
    char    s[20],val[20];
    int     range_signal;
    int     q;

/* Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/* Check for invalid arguments */
    if (asyc2_invalid_integer_range (config, 0, 4, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (gam, 0, 5, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (relat, 0, 2, -3) != 0)
       return asyc2_err;
    switch (gam)
        {
        case 0: /* AUTO or 500mA : +- 500mA */
        case 4:
            if (asyc2_invalid_real_range (ref, -500.0, 500.0, -4) != 0)
            return asyc2_err;
            break;
        case 1:   /* 500A : +- 500A */
            if (asyc2_invalid_real_range (ref, -0.5, 0.5, -4) != 0)
            return asyc2_err;
            break;
        case 2:   /* 5mA : +- 5mA  */
            if (asyc2_invalid_real_range (ref, -5, 5, -4) != 0)
            return asyc2_err;
            break;
        case 3:   /* 50mA : +- 50mA */
            if (asyc2_invalid_real_range (ref, -50, 50, -4) != 0)
            return asyc2_err;
            break;
        case 5:  /* 10A : +- 10A */
            if (asyc2_invalid_real_range (ref, -10000, 10000, -4) != 0)
            return asyc2_err;
            break;
        }
    if (asyc2_invalid_integer_range (zoom, 0, 1, -5) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -6) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -7) != 0)
       return asyc2_err;

/* Convert config into instrument parameter */
    if (config==1) config=4;
    else if (config==2) config=6;
    else if (config==3) config=2;
    else if (config==4) config=3;
    if (gam!=5) gam--;
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  500A not available on MX53                  */
/*  ZOOM and SURV not available on MX53 and MX55 */
    if(
       (((gam==0)||(gam==-1)) && (asyc2_instrument==MX53)) ||
       ((asyc2_instrument==MX53)||(asyc2_instrument==MX55))&&(zoom||sur)
      ){
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch    */
/*  Error 301: Wrong position of the rotary switch */
    if(
        (asyc2_commut<MA || asyc2_commut>uA) ||
        /* MX53 : Check game and Rotary switch compatibility */
        (((gam==5)^(asyc2_commut==A10)) && (asyc2_instrument==MX53)) ||
        (((gam==3)^(asyc2_commut==A500m)) && (asyc2_instrument==MX53)) ||
        (((gam==2)^(asyc2_commut==A50m)) && (asyc2_instrument==MX53 )) ||
        (((gam==1)^(asyc2_commut==A5m)) && (asyc2_instrument==MX53 ))
       ){
       asyc2_err=301;
       return asyc2_err;
    }

/*  Check compatibility of parameters (error 302) */
/*  Error 302 : Two arguments are incompatible    */
    if((sur!=0 && gam!=asyc2_range) ||
       (ref<0 && relat<0 && (config==PEAKP || config==AC || config==ACDC)) ||
       (ref>0 && relat<0 && config==PEAKM) ||
       ((config==PEAKP || config==PEAKM) && sur!=0)) {
       asyc2_err=302;
       return asyc2_err;
    }


/*  Set the instrument to current measurement */
    if(asyc2_ctrl_config!=config){
       Fmt(s,"%s<21%i\r",config);
       if(asyc2_send(s,4)!=0) return asyc2_err;
    }

/*  Set the range */
    if(asyc2_commut==MA && !(gam==AUTO &&
       (config==PEAKP || config==PEAKM)))
       asyc2_change_range(gam);

/*  Set the relative reference */
    if(relat<0 ){
       if(ref>=-0.5        && ref<=0.5   && (asyc2_commut==MA || uA))  range_signal=0;
       else if(ref>=-5.0   && ref<=5.0   && (asyc2_commut==MA || A5m))   range_signal=1;
       else if(ref>=-50.0  && ref<=50.0  && (asyc2_commut==MA || A50m))  range_signal=2;
       else if(ref>=-500.0 && ref<=500.0 && (asyc2_commut==MA || A500m)) range_signal=3;
       else if(asyc2_commut==A10)                                      range_signal=5;
       q=range_signal;
       if(ref<0){
          range_signal=range_signal+8;
          ref=ref*(-1);
       }
       ref=ref*1E+05;
       Fmt(val,"%s<%f[w10p0]",ref);

       if(asyc2_resol!=0) Fmt(s,"%s<23:%c%c%c%c%c%c\r",
       val[9-q],val[8-q],val[7-q],val[6-q],val[5-q],48+range_signal);
       else Fmt(s,"%s<23:%c%c%c%c0%c\r",
       val[8-q],val[7-q],val[6-q],val[5-q],48+range_signal);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,zoom);
    return asyc2_err;

}
/*=========================================================================*/
/* Function: dB                                                            */
/* Purpose:  This function sets the instrument to dBm measurement          */
/*=========================================================================*/
int asyc2_dB(int relat,double ref,int sur,int memo)
{
    char    s[20],val[20];
    int     range_signal;

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (relat, 0, 2, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_real_range (ref, -500.0, 500.0, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -3) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -4) != 0)
       return asyc2_err;

/* Convert into instrument parameter */
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  dB   not available on MX53  */
/*  SURV not available on MX55  */
    if ((asyc2_instrument==MX53)||
       ((asyc2_instrument==MX55)&&(sur)))
    {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch        */
/*  Error 301 : Wrong position of the rotary switch    */
    if(asyc2_commut!=VAC){
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to current measurement */
    if(asyc2_ctrl_config!=24)
    if(asyc2_send("218\r",4)!=0) return asyc2_err;

/*  Set the relative reference */
    if(relat<0){
       range_signal=1;
       if(ref<0){
          range_signal=9;
          ref=ref*(-1);
       }
       Fmt(val,"%s<%f[w6p2]",ref);
       if(asyc2_resol!=0) Fmt(s,"%s<23:%c%c%c%c%c%i\r",
        val[5],val[4],val[2],val[1],val[0],range_signal);
       else Fmt(s,"%s<23:%c%c%c%c0%i\r",
        val[4],val[2],val[1],val[0],range_signal);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,0);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Testline                                                      */
/* Purpose:  This function sets the instrument to testline measurement     */
/*=========================================================================*/
int asyc2_testline(int relat,double ref,int memo,int sur)
{
    char s[20],val[20];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (relat, 0, 2, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_real_range (ref, 0.0, 600.0, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -3) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -4) != 0)
       return asyc2_err;

/* Convert into instrument parameter */
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  Test Line on MX54 and MX56 only (not on MX53,MX55) */
    if((asyc2_instrument!=MX54 && asyc2_instrument!=MX56 || asyc2_benchtop==1)){
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch        */
/*  Error 301 : Wrong position of the rotary switch    */
/*  Not in VAC */
    if( asyc2_commut!=VAC ) {
       asyc2_err=301;
       return asyc2_err;
    }

/* Set the instrument to testline measurement */
    if(asyc2_ctrl_config!=25)
    if(asyc2_send("219\r",4)!=0) return asyc2_err;

/* Set the relative reference */
    if(relat<0){
       Fmt(val,"%s<%f[w6p2]",ref);
       Fmt(s,"%s<23:%c%c%c%c03\r",val[4],val[2],val[1],val[0]);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,0);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Power                                                         */
/* Purpose:  This function switch the instrument to resistive power        */
/*           measurement                                                   */
/*=========================================================================*/
int asyc2_power(int relat,int memo,int sur)
{
/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (relat, 0, 2, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -3) != 0)
       return asyc2_err;

/*  Convert into instrument parameter */
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  Power available on MX56 only (not on MX53,54,55) */
    if(asyc2_instrument!=MX56) {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch */
/*  Error 301 : Wrong position of the rotary switch    */
/*  not in VDC */
    if(asyc2_commut!=VDC) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to resistive power measurement */
    if(asyc2_ctrl_config!=26)
    if(asyc2_send("21:\r",4)!=0) return asyc2_err;

    asyc2_change_config(relat,sur,memo,0,0);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Frequency                                                     */
/* Purpose:  This function switch the instrument to frequency measurement    */
/*=========================================================================*/
int asyc2_frequency(int relat,int zoom,int memo,int sur)
{
/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (relat, 0, 2, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (zoom, 0, 1, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -3) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -4) != 0)
       return asyc2_err;

/*  Convert into instrument parameter */
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  ZOOM and SURV not available on MX53 and MX55 */
    if(
       ((asyc2_instrument==MX53)||(asyc2_instrument==MX55))&&(zoom||sur)
      ){
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch        */
/*  Error 301 : Wrong position of the rotary switch    */
/*  If MX53 or MX55 and not VAC */
/*  If MX54 or MX55 and not MV and not VDC and not (A/mA) and not 10A */
    if(
       (asyc2_commut!=VAC &&
         ((asyc2_instrument==MX53 || asyc2_instrument==MX55)
                                ||
          (asyc2_commut!=MV && asyc2_commut!=VDC && asyc2_commut!=MA && asyc2_commut!=A10 &&
           (asyc2_instrument==MX54 || asyc2_instrument==MX56)
          )
         )
       )
      ) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to frequency measurement */
    if(asyc2_ctrl_config!=41)
    if(asyc2_send("229\r",4)!=0) return asyc2_err;

    asyc2_change_config(relat,sur,memo,0,zoom);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Cycle                                                         */
/* Purpose:  This function switch the instrument to duty cycle measurement   */
/*=========================================================================*/
int asyc2_cycle(int sign,int relat,double ref,int memo,int sur)
{
    char    s[20],val[20];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (sign, 0, 1, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (relat, 0, 2, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_real_range (ref, 0.0, 100.0, -3) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -4) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -5) != 0)
       return asyc2_err;

/*  Convert into instrument parameter */
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  SURV not available on MX53 and MX55 */
    if (((asyc2_instrument==MX55)||(asyc2_instrument==MX53))&&(sur))
        {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch */
/*  Error 301 : Wrong position of the rotary switch    */
/*  If MX53 or MX55 and not VAC */
/*  If MX54 or MX55 and not MV and not VDC and not (A/mA) and not 10A */
    if(
       (asyc2_commut!=VAC &&
         ((asyc2_instrument==MX53 || asyc2_instrument==MX55)
                                ||
          (asyc2_commut!=MV && asyc2_commut!=VDC && asyc2_commut!=MA && asyc2_commut!=A10 &&
           (asyc2_instrument==MX54 || asyc2_instrument==MX56)
          )
         )
       )
      ) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to duty cycle measurement */
    if(asyc2_ctrl_config!=44+sign){
       Fmt(s,"%s<22%c\r",60+sign);
       if(asyc2_send(s,4)!=0) return asyc2_err;
    }

/*  Change the relative reference */
    if(relat<0 ) {
       Fmt(val,"%s<%f[w5p2]",ref);
       Fmt(s,"%s<23:%c%c%c%c00\r",val[4],val[3],val[1],val[0]);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,0);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Pulse Count                                                   */
/* Purpose:  This function switch the instrument to pulse count measurement  */
/*=========================================================================*/
int asyc2_pulse_count(int sign)
{
    char    s[20];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (sign, 0, 1, -1) != 0)
       return asyc2_err;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  Pulse count available on MX56 only (not on MX53,54,55,56) */
    if(asyc2_instrument!=MX56) {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch        */
/*  Error 301 : Wrong position of the rotary switch    */
/*  Not VAC or MV or VDC or mA or A10 */
    if (asyc2_commut!=VAC && asyc2_commut!=MV && asyc2_commut!=VDC &&
        asyc2_commut!=MA && asyc2_commut!=A10){
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to pulse count measurement */
    Fmt(s,"%s<22%c\r",62+sign);
    if(asyc2_send(s,4)!=0) return asyc2_err;

    return asyc2_err;
}
/*=========================================================================*/
/* Function: Chronometer                                                   */
/* Purpose:  This function switch the instrument to chrono measurement     */
/*=========================================================================*/
int asyc2_pulse_width(int sign,int memo,int sur)
{
    char    s[20];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (sign, 0, 1, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -3) != 0)
       return asyc2_err;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  Error 300 : Function not available         */
/*  Pulse width available on MX56 only (not on MX53,54,55,56) */
    if(asyc2_instrument!=MX56) {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch */
/*  Error 301 : Wrong position of the rotary switch    */
/*  Not VAC or MV or VDC or mA or A10 */
    if((asyc2_commut!=VAC && asyc2_commut!=MV && asyc2_commut!=VDC &&
        asyc2_commut!=MA && asyc2_commut!=A10)) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to chrono measurement */
    if(asyc2_ctrl_config!=42+sign){
       Fmt(s,"%s<22%c\r",58+sign);
       if(asyc2_send(s,4)!=0) return asyc2_err;
    }

    asyc2_change_config(0,sur,memo,0,0);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Resistance                                                    */
/* Purpose:  This function switch the instrument to resistance measurement   */
/*=========================================================================*/
int asyc2_resistance(int gam,int relat,double ref,int zoom,int memo,int sur)
{
    char s[20],val[20];
    int range_signal;

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (gam, 0, 6, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (relat, 0, 2, -2) != 0)
       return asyc2_err;
    switch (gam)
        {
        case 0: /* AUTO or 50MOHM : +- 50000 kOHM */
        case 6:
            if (asyc2_invalid_real_range (ref, -50000.0, 50000.0, -3) != 0)
            return asyc2_err;
            break;
        case 1:   /* 500OHM : +- 0.500 kOHM */
            if (asyc2_invalid_real_range (ref, -0.5, 0.5, -3) != 0)
            return asyc2_err;
            break;
        case 2:   /* 5kOHM : +- 5 kOHM  */
            if (asyc2_invalid_real_range (ref, -5, 5, -3) != 0)
            return asyc2_err;
            break;
        case 3:   /* 50kOHM : +- 50 kOHM */
            if (asyc2_invalid_real_range (ref, -50, 50, -3) != 0)
            return asyc2_err;
            break;
        case 4:  /* 500kOHM : +- 500 kOHM */
            if (asyc2_invalid_real_range (ref, -500, 500, -3) != 0)
            return asyc2_err;
            break;
        case 5:  /* 5MOHM : +- 5000 kOHM */
            if (asyc2_invalid_real_range (ref, -5000, 5000, -3) != 0)
            return asyc2_err;
            break;
        }
    if (asyc2_invalid_integer_range (zoom, 0, 1, -4) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -5) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -6) != 0)
       return asyc2_err;

/*  Convert into instrument parameter */
    gam--;
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  ZOOM and SURV not available on MX53 and MX55 */
    if (
        ((asyc2_instrument==MX55)||(asyc2_instrument==MX53))&&(zoom||sur))
        {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check compatibility of parameters             */
/*  Error 302 : Two arguments are incompatible    */
/*  Do not modify the range when SURV on */
    if(sur!=0 && gam!=asyc2_range) {
       asyc2_err=302;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch        */
/*  Error 301 : Wrong position of the rotary switch    */
/*  not OHM */
    if(asyc2_commut!=OHM) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to resistance measurement */
    if(asyc2_ctrl_config!=14)
        if(asyc2_send("20>\r",4)!=0) return asyc2_err;

/*  Set the range */
    asyc2_change_range(gam);

/*  Set the relative reference */
    if(relat<0 )
    {
       if(ref<=0.5)         range_signal=0;
       else if(ref<=5.0)    range_signal=1;
       else if(ref<=50.0)   range_signal=2;
       else if(ref<=500.0)  range_signal=3;
       else if(ref<=5000.0) range_signal=4;
       else                range_signal=5;
       ref=ref*1E+05;
       Fmt(val,"%s<%f[w10p0]",ref);
       if(asyc2_resol!=0) Fmt(s,"%s<23:%c%c%c%c%c%i\r",
        val[9-range_signal],val[8-range_signal],val[7-range_signal],
        val[6-range_signal],val[5-range_signal],range_signal);
       else Fmt(s,"%s<23:%c%c%c%c0%i\r",val[8-range_signal],val[7-range_signal],
        val[6-range_signal],val[5-range_signal],range_signal);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,zoom);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Continuity Test                                               */
/* Purpose:  This function set the instrument to continuity test           */
/*           measurement                                                   */
/*=========================================================================*/
int asyc2_continuity_test(int relat,double ref,int zoom,int memo,int sur)
{
    char s[20],val[20];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (relat, 0, 2, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_real_range (ref, 0.0, 500.0, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (zoom, 0, 1, -3) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -4) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -5) != 0)
       return asyc2_err;

/*  Convert into instrument parameter */
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  ZOOM and SURV not available on MX53 and MX55 */
    if (
        ((asyc2_instrument==MX53)||(asyc2_instrument==MX55))&&(zoom||sur))
        {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch */
/*  Error 301 : Wrong position of the rotary switch    */
/*  Not OHM  */
    if(asyc2_commut!=OHM) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to continuity test measurement */
    if(asyc2_ctrl_config!=11)
        if(asyc2_send("20;\r",4)!=0) return asyc2_err;

/*  Set the relative reference */
    if(relat<0){
       Fmt(val,"%s<%f[w6p2]",ref);
       if(asyc2_resol!=0) Fmt(s,"%s<23:%c%c%c%c%c0\r",val[5],val[4],val[2],
            val[1],val[0]);
       else Fmt(s,"%s<23:%c%c%c%c00\r",val[4],val[2],val[1],val[0]);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,zoom);
    return asyc2_err;
 }
/*=========================================================================*/
/* Function: Capacitance                                                   */
/* Purpose:  This function set the instrument to capacitance measurement   */
/*=========================================================================*/
int asyc2_capacitance(int gam,int relat,double ref,int zoom,int memo,int sur)
{
    char s[20],val[20];
    int range_signal;

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (gam, 0, 7, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (relat, 0, 2, -2) != 0)
       return asyc2_err;
    switch (gam)
        {
        case 0: /* AUTO or 50mF : +- 50000 F*/
        case 7:
            if (asyc2_invalid_real_range (ref, -50000.0, 50000.0, -3) != 0)
            return asyc2_err;
            break;
        case 1:   /* 50nF : +- 0.05 F */
            if (asyc2_invalid_real_range (ref, -0.05, 0.05, -3) != 0)
            return asyc2_err;
            break;
        case 2:   /* 500nF : +- 0.5 F  */
            if (asyc2_invalid_real_range (ref, -0.5, 0.5, -3) != 0)
            return asyc2_err;
            break;
        case 3:   /* 5F : +- 5F */
            if (asyc2_invalid_real_range (ref, -5, 5, -3) != 0)
            return asyc2_err;
            break;
        case 4:  /* 50F : +- 50F */
            if (asyc2_invalid_real_range (ref, -50, 50, -3) != 0)
            return asyc2_err;
            break;
        case 5:  /* 500F : +- 500F */
            if (asyc2_invalid_real_range (ref, -500, 500, -3) != 0)
            return asyc2_err;
            break;
        case 6:  /* 5000F : +- 5000F */
            if (asyc2_invalid_real_range (ref, -5000, 5000, -3) != 0)
            return asyc2_err;
            break;
        }
    if (asyc2_invalid_integer_range (zoom, 0, 1, -4) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -5) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -6) != 0)
       return asyc2_err;

/*  Convert into instrument parameter */
    gam--;
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  ZOOM and SURV not available on MX53 and MX55 */
    if (
        ((asyc2_instrument==MX53)||(asyc2_instrument==MX55))&&(zoom||sur))
        {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check compatibility of parameters             */
/*  Error 302 : Two arguments are incompatible    */
/*  Do not modify Range when SURV on */
    if(sur!=0 && gam!=asyc2_range){
       asyc2_err=302;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch */
/*  Error 301 : Wrong position of the rotary switch    */
/*  not CAPA */
    if(asyc2_commut!=CAPA) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to capacitance measurement */
    if(asyc2_ctrl_config!=15)
    if(asyc2_send("20?\r",4)!=0) return asyc2_err;

/*  Set the range */
    asyc2_change_range(gam);

/*  Set the relative reference */
    if(relat<0 ) {
       if(ref<=0.05)        range_signal=0;
       else if(ref<=0.5)    range_signal=1;
       else if(ref<=5.0)    range_signal=2;
       else if(ref<=50.0)   range_signal=3;
       else if(ref<=500.0)  range_signal=4;
       else if(ref<=5000.0) range_signal=5;
       else                range_signal=6;
       ref=ref*1E+05;
       Fmt(val,"%s<%f[w10p0]",ref);
       Fmt(s,"%s<23:%c%c%c%c0%i\r",val[9-range_signal],val[8-range_signal],
            val[7-range_signal],val[6-range_signal],range_signal);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,zoom);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Diode Voltage                                                 */
/* Purpose:  This function switch the instrument to diode voltage measurement*/
/*=========================================================================*/
int asyc2_diode(int relat,double ref,int memo,int sur)
{
    char s[20],val[20];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (relat, 0, 2, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_real_range (ref, 0.0, 2.0, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -3) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -4) != 0)
       return asyc2_err;

/*  Convert into instrument parameter */
    if (relat==2) relat=-1;

  /*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  ZOOM and SURV not available on MX53 and MX55 */
    if (
        ((asyc2_instrument==MX53)||(asyc2_instrument==MX55))&&(sur))
        {
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch */
/*  Error 301 : Wrong position of the rotary switch    */
/*  Not on CAPA */
    if(asyc2_commut!=CAPA) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to diode voltage measurement */
    if(asyc2_ctrl_config!=13)
    if(asyc2_send("20=\r",4)!=0) return asyc2_err;

/*  Set the relative reference */
    if(relat<0){
       Fmt(val,"%s<%f[w5p3]",ref);
       Fmt(s,"%s<23:%c%c%c%c01\r",val[4],val[3],val[2],val[0]);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,0);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Temperature                                                   */
/* Purpose:  This function switch the instrument to chrono measurement     */
/*=========================================================================*/
int asyc2_temperature(int deg,int sensor,int memo,int sur,int relat,double ref)
{
    char s[20],val[20];
    int range_signal;

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (deg, 0, 1, -1) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sensor, 0, 1, -2) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (memo, 0, 1, -3) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (sur, 0, 1, -4) != 0)
       return asyc2_err;
    if (asyc2_invalid_integer_range (relat, 0, 2, -5) != 0)
       return asyc2_err;
    if ( ref<0 ||
         (ref>=500.0 && sensor==1 && relat<0) ||
         (ref>=5000.0 && sensor==0 && relat<0)){
       asyc2_err=-6;
       return asyc2_err;
    }

/*  Convert into instrument parameter */
    if (relat==2) relat=-1;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
/*  TEMP only on MX54 */
    if(asyc2_instrument!=MX54){
       asyc2_err=300;
       return asyc2_err;
    }

/*  Check for the position of the rotary switch        */
/*  Error 301 : Wrong position of the rotary switch    */
/*  Not on TEMP */
    if(asyc2_commut!=TEMP) {
       asyc2_err=301;
       return asyc2_err;
    }

/*  Set the instrument to temperature measurement */
    if(asyc2_ctrl_config!=10)
    if(asyc2_send("20:\r",4)!=0) return asyc2_err;

/*  Set units */
    if((asyc2_degre==0 && deg!=0) || (asyc2_degre==1 && deg==0))
    if(asyc2_send("239\r",4)!=0) return asyc2_err;

/*  Set the sensor */
    if((asyc2_sensorpt==0 && sensor!=0) || (asyc2_sensorpt==1 && sensor==0))
    if(asyc2_send("23=\r",4)!=0) return asyc2_err;

/*  Set the relative reference */
    if(relat<0) {
       range_signal=0;
       if(sensor==0) range_signal=1;
       ref=ref*1E+02;
       Fmt(val,"%s<%f[w6p0]",ref);
       Fmt(s,"%s<23:%c%c%c%c%c%i\r",val[5-range_signal],val[4-range_signal],
        val[3-range_signal],val[2-range_signal],val[1-range_signal],0);
       if(asyc2_send(s,10)!=0) return asyc2_err;
    }

    asyc2_change_config(relat,sur,memo,0,0);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Resolution                                                    */
/* Purpose:  This function switch the resolution of instrument               */
/*=========================================================================*/
int asyc2_resolution(int resol)
{
/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (resol, 0, 1, -1) != 0)
       return asyc2_err;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Check compatibility of parameters             */
/*  Error 302 : Two arguments are incompatible    */
    if(
       ((resol==1)&&                   /* high resolution disabled when */
            (
                (asyc2_ctrl_config==25)||  /* Test Line */
                (asyc2_ctrl_config==44)||  /* DC +      */
                (asyc2_ctrl_config==45)||  /* DC -      */
                (asyc2_ctrl_config==15)||  /* Capa      */
                (asyc2_ctrl_config==13)||  /* Diode     */
                (asyc2_ctrl_config== 2)||  /* PK+ (A)   */
                (asyc2_ctrl_config== 3)||  /* PK- (A)   */
                (asyc2_ctrl_config==18)||  /* PK+ (V)   */
                (asyc2_ctrl_config==19)||  /* PK- (V)   */
                ((asyc2_instrument==MX53)&&
                 ((asyc2_ctrl_config== 4)||  /* AC (A)  */
                  (asyc2_ctrl_config==20)||  /* AC (V)  */
                  (asyc2_ctrl_config== 6)||  /* AC+DC (A)  */
                  (asyc2_ctrl_config==22)    /* AC+DC (V)  */
                 )
                )
            )
       )
       ||
       ((resol==0)&&                   /* Low resolution disabled when */
            (
                (asyc2_ctrl_config==24)||  /* dB        */
                (asyc2_ctrl_config==41)||  /* Frequency */
                (asyc2_ctrl_config==46)||  /* Counter + */
                (asyc2_ctrl_config==47)||  /* Counter - */
                (asyc2_ctrl_config==42)||  /* P Width + */
                (asyc2_ctrl_config==43)||  /* P Width - */
                (asyc2_ctrl_config==26)||  /* Power     */
                (asyc2_ctrl_config==10)    /* Temp      */
            )
       )
      ){
       asyc2_err=302;
       return asyc2_err;
    }

/*  Set the resolution of the instrument */
    if((asyc2_resol==0 && resol!=0) || (asyc2_resol==1 && resol==0))
    if(asyc2_send("247\r",4)!=0) return asyc2_err;
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Rejection                                                     */
/* Purpose:  This function switch the rejection of the instrument            */
/*=========================================================================*/
int asyc2_rejection(int reject)
{
/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (reject, 0, 1, -1) != 0)
       return asyc2_err;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Set the rejection of the instrument */
    if((asyc2_reject==0 && reject!=0) || (asyc2_reject==1 && reject==0))
    if(asyc2_send("245\r",4)!=0) return asyc2_err;
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Input Impedance                                               */
/* Purpose:  This function switch the input impedance for measurement in   */
/*           mV range                                                      */
/*=========================================================================*/
int asyc2_input_impedance(int ZmV)
{
/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (ZmV, 0, 1, -1) != 0)
    return asyc2_err;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Set the input impedance of the instrument */
    if((asyc2_Ze==0 && ZmV!=0) || (asyc2_Ze==1 && ZmV==0))
    if(asyc2_send("246\r",4)!=0) return asyc2_err;

    return asyc2_err;
}
/*=========================================================================*/
/* Function: LCD                                                           */
/* Purpose:  This function switch the lcd to full display or not           */
/*=========================================================================*/
int asyc2_lcd(int control)
{
    char s[20];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (control, 0, 1, -1) != 0)
       return asyc2_err;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Set lcd mode */
    Fmt(s,"%s<37%i\r",control);
    if(asyc2_send(s,4)!=0) return asyc2_err;
    return asyc2_err;
}
/*=========================================================================*/
/* Function: OHM reference                                                 */
/* Purpose:  This function adjusts the ohm value reference for dB          */
/*           measurement or resistive power measurement                    */
/*=========================================================================*/
int asyc2_ref_ohm(int ohm_val)
{
    char s[20],chaine[20];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (ohm_val, 0, 9999, -1) != 0)
       return asyc2_err;

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Adjust the new ohm value if a change has occured */
    if(ohm_val!=asyc2_ohm_value){
       asyc2_ohm_value=ohm_val;
       Fmt(chaine,"%s<%i[w4p0]",asyc2_ohm_value);
       Fmt(s,"%s<23;%c%c%c%c\r",chaine[3],chaine[2],chaine[1],chaine[0]);
       if(asyc2_send(s,8)!=0) return asyc2_err;
    }
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Timer                                                         */
/* Purpose:  This function adjust the timer value for repetitive           */
/*           measurement                                                   */
/*=========================================================================*/
int asyc2_timer(char *cadence)
{
    char s[20],chaine[20];
    long duree;

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (Scan(cadence,"%s>%i[b4]%s",&duree,s)!=1) {
       asyc2_err=-1;
       return asyc2_err;
    }
    Fmt(s,"%s<%i[b4w5p0]",duree);
    if (s[1]>0x35 ||
        s[3]>0x35 ||
    duree>95959)  {
       asyc2_err=-1;
       return asyc2_err;
    }

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

/*  Adjust the new timer value if a change has occured */
    if(duree!=asyc2_clock) {
       asyc2_clock=duree;
       Fmt(chaine,"%s<23%<%c%c%c%c%c\r",s[4],s[3],s[2],s[1],s[0]);
       if(asyc2_send(chaine,9)!=0) return asyc2_err;
    }
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Status                                                        */
/* Purpose:  This function checks the status of the instrument and         */
/*           initializes static variables to give informations to others   */
/*           functions.                                                    */
/*=========================================================================*/
int asyc2_status(int *instrument,int *version,long *date,
                 int *resol,int *reject,int *commut,
                 int *ctrl_config,int *range,int *Ze,
                 int *ohm_value,long *clock,int *temp,
                 int *fuses,int *batt)
{
    char chaine[20];
    char chaine2[6];

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

    FillBytes (chaine,  0, 15, 0);
    FillBytes (chaine2, 0,  5, 0);

/*  Ask for status */
    asyc2_err=0;
    if(asyc2_write_data("35\r",3)!=0)       return asyc2_err;
    if(asyc2_read_data(chaine,15,13)!=0)    return asyc2_err;

    if (rscnt==10) asyc2_version=0;
    if (rscnt==14) asyc2_version=1;

/* Check if the right instrument is connected */
    if(chaine[0]-0x30!=asyc2_instrument)  {
       asyc2_err=223;
       return asyc2_err;
    }

/*  Read the rotary switch position (0..15) */
    asyc2_commut=chaine[1]-0x30;

 /*  Read the configuration of the measurement */
    asyc2_ctrl_config=(chaine[2]-0x30)*16+chaine[3]-0x30;
    /* If current and DC,DC Pk+,DC Pk-,RMS,RMS AC+DC substract 16 to */
    /* obtain codes 0,2,3,4,6 for asyc2_ctrl_config   */
    if((asyc2_commut>7) && (asyc2_commut<14) && (chaine[3]< 0x37))
       asyc2_ctrl_config=asyc2_ctrl_config-16;

/*  Read the range of the measurement */
    asyc2_range   =  chaine[5]-0x30;

/*  Read LRHD,MEM,SURV and REL mode */
    asyc2_lrhd    =  chaine[6]&0x01;
    asyc2_mem     = (chaine[6]>>1)&0x01;
    asyc2_surv    = (chaine[6]>>2)&0x01;
    asyc2_rel     = (chaine[6]>>3)&0x01;

/*  Read the manual/AUTO ranging mode */
    asyc2_ranging =  chaine[7]&0x01;
/*  Read the normal/calibration mode */
    asyc2_type    = (chaine[7]>>1)&0x01;
/*  Read the ZOOM mode /(off/on) */
    asyc2_zoom    = (chaine[7]>>2)&0x01;

/*  Read Input Impedance (GOHM/10 MOHM) */
    asyc2_Ze      =  chaine[8]&0x01;
/*  Read Rejection mode (50Hz/60Hz)     */
    asyc2_reject  = (chaine[8]>>1)&0x01;
/*  Read sensor (pt 1000/pt 100) */
    asyc2_sensorpt = (chaine[8]>>2)&0x01;
/*  Read degre unit (C/F) */
    asyc2_degre   = (chaine[8]>>3)&0x01;

/*  Read Resolution (low/High) */
    asyc2_resol   =  chaine[9]&0x01;
/*  Read Fuse 1 (On/Off) */
    asyc2_fuse1   = (chaine[9]>>1)&0x01;
/*  Read Fuse 2 (On/Off) */
    asyc2_fuse2   = (chaine[9]>>2)&0x01;
/*  Read Battery (On/Off) */
    asyc2_batt    = (chaine[9]>>3)&0x01;

/*  If version V2, read Battery Level */
    if (asyc2_version == 1)
       asyc2_sbat = (chaine[10]-0x30)*1000 + (chaine2[0]-0x30)*100 +
            (chaine2[1]-0x30)*10   + (chaine2[2]-0x30);

    *instrument =   asyc2_instrument;
    *version    =   asyc2_version;
    *date       =   asyc2_ldate;
    *resol      =   asyc2_resol;
    *reject     =   asyc2_reject;
    *commut     =   asyc2_commut;
    *ctrl_config=   asyc2_ctrl_config;
    *range      =   asyc2_range;
    *Ze         =   asyc2_Ze;
    *ohm_value  =   asyc2_ohm_value;
    *clock      =   asyc2_clock;
    if (asyc2_degre==0)
        if (asyc2_sensorpt==0) *temp=00;
        else                   *temp=01;
    else
        if (asyc2_sensorpt==0) *temp=10;
        else                   *temp=11;

    if (asyc2_fuse2==0)
        if (asyc2_fuse1==0)    *fuses=11;
        else                   *fuses=10;
    else
        if (asyc2_fuse1==0)    *fuses=01;
        else                   *fuses=00;

    *batt       =   asyc2_batt;

    return asyc2_err;
}
/*=========================================================================*/
/* Function: Data                                                          */
/* Purpose:  This function configure the instrument for MIN, MAX, AVG, next, */
/*           previous or repetitive measurement. The result is in          */
/*           asyc2_dmeasurement, asyc2_smeasurement and asyc2_unit.        */
/*=========================================================================*/
int asyc2_data(int measure, double *result, int *type)
{
    char s[20],chaine[20];
    double multi;
    int code,gam;

/*  Erase string s */
    FillBytes (s, 0, 16, 0);

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (measure, 0, 5, -1) != 0)
       return asyc2_err;

/*  Check for the availability of the function */
/*  Error 300 : Function not available         */
    /* MIN,MAX,AVG not available on MX53 and MX55 */
    if (
        ((asyc2_instrument==MX53)||(asyc2_instrument==MX55))&&(measure<3))
        {
        asyc2_err=300;
        return asyc2_err;
    }

    switch(measure)  {
    /* MIN, MAX or AVG measurement */

        case 0:
        case 1:
        case 2: Fmt(s,"%s<38%i\r",measure);
                if(asyc2_write_data(s,4)!=0)    return asyc2_err;
                if (asyc2_read_data(s,11,13)!=0) return asyc2_err;

                /* Test overflow (if bit 3 of s[6] ==1) */
                if((s[6]&0x08)==1) {
                    asyc2_dmeasure=99999;
                    *result= asyc2_dmeasure;
                    asyc2_err=303;
                    return asyc2_err;
                }
                /* Arrange the string MSD ..LSD and store the result */
                /* in the double asyc2_dmeasure                      */
                Fmt(chaine,"%s<%c%c%c%c%c",s[4],s[3],s[2],s[1],s[0]);
                Scan(chaine,"%s>%f",&asyc2_dmeasure);

                /* Check sign (<0 if bit 3 of s[5] == 1) */
                if((s[5]&0x08)==1) {
                    asyc2_dmeasure=asyc2_dmeasure*(-1);
                    /* Reset bit 3 */
                    s[5]=s[5]&0xF7;
                }

                /* Test resolution (50000 if bit 0 of s[6] == 1, else 5000 */
                if((s[6]&0x01)==1) asyc2_dmeasure=asyc2_dmeasure*0.1;

                s[7]=s[7]-0x30;

                /* Read Range of measurement */
                gam=s[5]-0x30;
                /* Read Type of measurement */
                code=(s[9]-0x30)*16+(s[8]-0x30);

                /* Set the decimal point according to Type and Range */
                /* of measurement (*1,*0.1,*0.01 or *0.001)          */

                /* if audio power Range 4:  multiply by 10 */
                if(code==26 && gam==4) multi=1.0E+01;

                /* if DC,DC Pk+,DC Pk-,RMS,RMS AC+RC Range 4  */
                /* or audio power Range 3                     */
                /* or Capacity Range 5                        */
                /* or Temperature Range 1                     */
                /* multiply by 1                              */
                if( (code>=16 && code<=22 && gam==4) ||
                    (code==26 && gam==3) ||
                    (code==15 && gam==5) ||
                    (code==10 && gam==1))
                    multi=1.0E-00;

                /* if DC,DC Pk+,DC Pk-,RMS,RMS AC+RC Range 0 or 3  (500mV or 500V) */
                /* or frequency Range 2 or 5                       */
                /* or audio power Range 2 (50V)                    */
                /* or Temperature Range 1                          */
                /* or P Width+ P Width- Range 0                    */
                /* or OHM,Continuity Range 0 or 3 (500 or 500K OHM)*/
                /* or Capacity Range 1 or 4 (500nF or 500F)       */
                /* or dB,Test Line                                 */
                /* multiply by 0.1                                 */
                if( (code>=16 && code<=22 && (gam==0 || gam==3)) ||
                    (code==41 && (gam==2 || gam==5)) ||
                    (code==26 && gam==2) ||
                    (code==10 && gam==1) ||
                    ((code==42 || code==43) && gam==0) ||
                    ((code==14 || code==11) && (gam==0 || gam==3)) ||
                    (code==15 && (gam==1 || gam==4)) ||
                    (code==24 || code==25))
                    multi=1.0E-01;

                /* if DC,DC Pk+,DC Pk-,RMS,RMS AC+RC Range 2 or 5  */
                /* or frequency Range 2 or 5                       */
                /* or audio power Range 2 (50V)                    */
                /* or Temperature Range 1                          */
                /* or P Width+ P Width- Range 0                    */
                /* or OHM,Continuity Range 0 or 3 (500 or 500K OHM)*/
                /* or Capacity Range 1 or 4 (500nF or 500F)       */
                /* or dB,Test Line                                 */
                /* multiply by 0.01                                */
                if( (code>=16 && code<=22 && (gam==2 || gam==5)) ||
                    (code==41 && (gam==1 || gam==4)) ||
                    (code==15 && (gam==0 || gam==3 || gam==6)) ||
                    (code==44 || code==45) ||
                    ((code==42 || code==43) && gam==2) ||
                    (code==14 && (gam==2 || gam==5)) ||
                    (code==26 && gam==1))
                    multi=1.0E-02;

                /* if DC,DC Pk+,DC Pk-,RMS,RMS AC+RC Range 1 (5V)  */
                /* or frequency Range 0 or 3                       */
                /* or audio power Range 0 (500mV)                  */
                /* or P Width+ P Width- Range 1                    */
                /* or OHM Range 1 or 4 (5k or 5M OHM)              */
                /* or Capacity Range 2 (5F)                       */
                /* or Diode                                        */
                /* multiply by 0.001                               */
                if( (code>=16 && code<=22 && gam==1) ||
                    (code==41 && (gam==0 || gam==3)) ||
                    (code==26 && gam==0) ||
                    ((code==42 || code==43) && gam==1) ||
                    (code==14 && (gam==1 || gam==4)) ||
                    (code==15 && gam==2) ||
                    (code==13))
                    multi=1.0E-03;

                asyc2_dmeasure=asyc2_dmeasure*multi;
                /* Test unit (n,,m,k,M) */
                asyc2_coef=1;
                switch(code){
                    /* OHM */
                    case 14:
                        /* OHM */
                        if (gam==0)                 asyc2_coef=1;
                        /* MOHM */
                        else if(gam==4 || gam==5)   asyc2_coef=1e+6;
                        /* kOHM */
                        else                        asyc2_coef=1e+3;
                        break;
                    /* Capacity */
                    case 15:
                        /* mF */
                        if(gam==6)                  asyc2_coef=1e-3;
                        /* nF */
                        else if(gam==0 || gam==1)   asyc2_coef=1e-9;
                        /* F */
                        else                        asyc2_coef=1e-6;
                        break;
                    /* DC,AC,AC+DC */
                    case 16:
                    case 20:
                    case 22:
                        /* Voltage */
                        if(s[7]<7) {
                            if(gam==0)          asyc2_coef=1e-3;
                        }
                        /* Current */
                        else {
                            code=code-16;
                            /* A  */
                            if(gam==0)          asyc2_coef=1e-6;
                            /* mA  */
                            else if (gam!=5)    asyc2_coef=1e-3;
                        }
                        break;
                    /* Frequency */
                    case 41:
                        /* kHz */
                        if(gam>=3)              asyc2_coef=1e+3;
                        break;
                    /* P Width+,P Width - */
                    case 42:
                    case 43:
                        /* mS */
                        if(gam==0)              asyc2_coef=1e-3;
                        break;
                }
                Fmt(asyc2_smeasure,"%s<%f[w7]",asyc2_dmeasure);
                asyc2_dmeasure=asyc2_dmeasure*asyc2_coef;
                *result= asyc2_dmeasure;
                *type=code;
                break;

        /* Last (3) or Next (4) measurement */
        case 3:
        case 4:
                /*  Read the configuration of the asyc2 */
                if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0)
                        return asyc2_err;
                if(measure==4) {
                    /* Set Timeout to a large value */
                    /* for waiting measurements on  */
                    /* RS232 line .                 */
                    SetComTime(port,40000.0);
                }
                Fmt(s,"%s<3%i\r",measure-3);
                if(asyc2_write_data(s,3)!=0) return asyc2_err;
                if (measure==4) {
                    /* Wait for the result */
                    while (GetInQLen(port)==0){}
                }
                /* Read Measurement */
                asyc2_decode();
                /* Set Time out to 1.0 seconde */
                if(measure==4) SetComTime(port,1.0);
                *type=asyc2_ctrl_config;
                *result= asyc2_dmeasure;
                break;
    }
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the port for the instrument module and   */
/*           sets the port to zero.                                        */
/*=========================================================================*/
int asyc2_close (void)
{
    asyc2_err=0;

/*  Check for device closed */
    if (asyc2_device_closed())
    return asyc2_err;

/*  Wait for the end of the last send */
    while (GetOutQLen (port)!=0) {}

/*  Close the COM port. If error, set asyc2_err = 221 */
    CloseCom (port);
    if (rs232err != 0) {
       asyc2_err = 221;
       return asyc2_err;
    }

    port = 0;
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Decode                                                        */
/* Purpose:  This function reads the last measurement from the RS232 and   */
/*           initializes asyc2_dmesure, asyc2_smeasure                     */
/*=========================================================================*/
int asyc2_decode(void)
{
    char s[20],chaine[20];

/*  Read the last measurement */
    FillBytes (s, 0, 16, 0);
    if (asyc2_read_data(s,16,13)!=0) return asyc2_err;

/*  Test if the measurement is overflow */
    if (s[6]=='X') {
       Fmt(asyc2_smeasure,"%s< ");
       asyc2_dmeasure=99999;
       asyc2_err=303;
    }

/*  Read measurement */
    if(s[1]==' ')
          Fmt(chaine,"%s<%c%c%c%c%c%c",  s[0],     s[2],s[3],s[4],s[5],s[6]);
    else  Fmt(chaine,"%s<%c%c%c%c%c%c%c",s[0],s[1],s[2],s[3],s[4],s[5],s[6]);
    Scan(chaine,"%s>%f",&asyc2_dmeasure);
    /* set asyc2_dmesure > 0 */
    if (asyc2_dmeasure==0.0) asyc2_dmeasure=0.0;
    Fmt(asyc2_smeasure,"%s<%c%c%c%c%c%c%c",s[0],s[1],s[2],s[3],s[4],s[5],s[6]);

/*  Read unity and adjust asyc2_dmeasure */
    asyc2_coef=1;
    if(s[7]=='n') asyc2_coef=1e-9;
    if(s[7]=='u') asyc2_coef=1e-6;
    if(s[7]=='m') asyc2_coef=1e-3;
    if(s[7]=='k') asyc2_coef=1e+3;
    if(s[7]=='M') asyc2_coef=1e+6;
    asyc2_dmeasure=asyc2_dmeasure*asyc2_coef;

    return asyc2_err;
}
/*=========================================================================*/
/* Function: Change Range                                                  */
/* Purpose:  This function change systematically the range of measurement. */
/*=========================================================================*/
int asyc2_change_range(int gam)
{
    char s[20];

/*  Check for the status of the instrument */
    if (asyc2_status(&i,&i,&l,&i,&i,&i,&i,&i,&i,&i,&l,&i,&i,&i)!=0) return asyc2_err;

    /*  If AUTO then send "24::\r" else send "24:gam\r" */
    if(gam==-1) Fmt(s,"%s<24::\r");
    else        Fmt(s,"%s<24:%i\r",gam);
    asyc2_send(s,5);

    return asyc2_err;
}
/*=========================================================================*/
/* Function: Change Config                                                 */
/* Purpose:  This function change REL, SURV, MEM, LRHD and ZOOM mode.      */
/*=========================================================================*/
int asyc2_change_config(int relat,int sur,int memo,int lrh,int zoo)
{
    double  result;
    int     type;

/*  Check if measurement overflow (Error 303) */
    if (asyc2_data(3,&result,&type)!=0) return asyc2_err;

/*  Change REL mode */
    /* If Ref=Last value and ASYC2 is already in REL mode */
    /* set REL Off first (toggle) */
    if ((relat ==1)&&(asyc2_rel==1)) {
       if(asyc2_send("233\r",4)!=0) return asyc2_err;
    }
    if((relat==1 ) || (relat==0 && asyc2_rel==1)) {
       asyc2_surv=0;
       asyc2_mem=0;
       asyc2_lrhd=0;
       if(asyc2_send("233\r",4)!=0) return asyc2_err;
    }

/*  Change SURV mode */
    if(((sur==0 && asyc2_surv==1) || (sur!=0 && asyc2_surv==0)) &&
       (asyc2_instrument==MX54 || asyc2_instrument==MX56)) {
       /* Clear asyc2_mem and asyc2_lrhd (SURV disables MEM and LRHD) */
       asyc2_mem=0;
       asyc2_lrhd=0;
       if(asyc2_send("231\r",4)!=0) return asyc2_err;
    }

/*  Change MEM mode */
    if((memo==0 && asyc2_mem==1) || (memo!=0 && asyc2_mem==0))
    if(asyc2_send("235\r",4)!=0) return asyc2_err;

/*  Change LRHD mode */
    if((lrh!=0 && asyc2_lrhd==0) || (lrh==0 && asyc2_lrhd==1))
    if(asyc2_send("232\r",4)!=0) return asyc2_err;

/*  Change ZOOM mode */
    if(((zoo!=0 && asyc2_zoom==0) ||
       (zoo==0 && asyc2_zoom==1 && asyc2_ctrl_config!=25)) &&
       (asyc2_instrument==MX54 || asyc2_instrument==MX56))
       if(asyc2_send("236\r",4)!=0) return asyc2_err;

    return asyc2_err;
}
/*=========================================================================*/
/* Function: Read Eeprom                                                   */
/* Purpose:  This function read a data from the eeprom                     */
/*=========================================================================*/
int asyc2_read_eeprom(int address)
{
    char s[20],chaine[20];
    int i;

/*  Check for device closed */
    if (asyc2_device_closed()) return asyc2_err;

/*  Check for invalid arguments */
    if (asyc2_invalid_integer_range (address, 0, 63, -1) != 0)
       return asyc2_err;

/*  Read the eeprom */
    Fmt(s,"%s<%x[w2]",address);
    if(s[1]>96) s[1]=s[1]-39;
    Fmt(chaine,"%s<10%c%c\r",s[0],s[1]);
    if(asyc2_write_data(chaine,5)!=0) return asyc2_err;
    if(asyc2_read_data(s,5,13)!=0)    return asyc2_err;

    for(i=0;i<4;i++) if(s[i]>0x39) s[i]=s[i]+7;
    Fmt(asyc2_code_eeprom,"%s<%c%c%c%c",s[0],s[1],s[2],s[3]);
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int asyc2_invalid_integer_range (int val, int min, int max, int err_code)
{
  if ((val < min) || (val > max)) {
    asyc2_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int asyc2_invalid_real_range (double val, double min, double max, int err_code)
{
  if ((val < min) || (val > max)) {
    asyc2_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int asyc2_read_data (char *buf, int cnt, int term)
{
    rscnt = ComRdTerm(port, buf, cnt, term);
    if (rs232err != 0)
        asyc2_err = 231;
    else
        asyc2_err = 0;
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int asyc2_write_data(char *buf,int cnt)
{
    /* This delay is necessary for the RS232 ASYC2 hardware */
    Delay(0.2);
    FlushInQ(port);
    FlushOutQ(port);
    rscnt = ComWrt(port,buf,cnt);
    if (rs232err !=0)
        asyc2_err=230;
    else
        asyc2_err=0;
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Send                                                          */
/* Purpose:  This function writes a buffer of data to the instrument and   */
/*           waits for an acknoledge.                                      */
/*=========================================================================*/
int asyc2_send(char *buf,int cnt)
{
    /* This delay is necessary for the RS232 ASYC2 hardware */
    Delay(0.2);
    FlushInQ(port);
    FlushOutQ(port);
    rscnt = ComWrt(port,buf,cnt);
    asyc2_err=0;
    if (rs232err !=0)
        {
        asyc2_err=230;
        return asyc2_err;
        }
    /* Wait for ACK (if Timeout return 240 */
    if(ComRdByte(port)!=6) asyc2_err=240;
    return asyc2_err;
}
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a -1 is      */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int asyc2_device_closed (void)
{
    if (port == 0)  {
       asyc2_err = 232;
       return -1;
    }
    return 0;
}
